/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.unsafe.map;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.apache.spark.unsafe.PlatformDependent;
import org.apache.spark.unsafe.array.ByteArrayMethods;
import org.apache.spark.unsafe.array.LongArray;
import org.apache.spark.unsafe.bitset.BitSet;
import org.apache.spark.unsafe.hash.Murmur3_x86_32;
import org.apache.spark.unsafe.map.HashMapGrowthStrategy;
import org.apache.spark.unsafe.memory.MemoryBlock;
import org.apache.spark.unsafe.memory.MemoryLocation;
import org.apache.spark.unsafe.memory.TaskMemoryManager;
import org.spark-project.guava.annotations.VisibleForTesting;

public final class BytesToBytesMap {
    private static final Murmur3_x86_32 HASHER = new Murmur3_x86_32(0);
    private static final HashMapGrowthStrategy growthStrategy = HashMapGrowthStrategy.DOUBLING;
    private static final int END_OF_PAGE_MARKER = -1;
    private final TaskMemoryManager memoryManager;
    private final List<MemoryBlock> dataPages = new LinkedList<MemoryBlock>();
    private MemoryBlock currentDataPage = null;
    private long pageCursor = 0L;
    private static final long PAGE_SIZE_BYTES = 0x4000000L;
    @VisibleForTesting
    static final int MAX_CAPACITY = 0x20000000;
    private LongArray longArray;
    private BitSet bitset;
    private final double loadFactor;
    private int size;
    private int growthThreshold;
    private int mask;
    private final Location loc;
    private final boolean enablePerfMetrics;
    private long timeSpentResizingNs = 0L;
    private long numProbes = 0L;
    private long numKeyLookups = 0L;
    private long numHashCollisions = 0L;

    public BytesToBytesMap(TaskMemoryManager memoryManager, int initialCapacity, double loadFactor, boolean enablePerfMetrics) {
        this.memoryManager = memoryManager;
        this.loadFactor = loadFactor;
        this.loc = new Location();
        this.enablePerfMetrics = enablePerfMetrics;
        if (initialCapacity <= 0) {
            throw new IllegalArgumentException("Initial capacity must be greater than 0");
        }
        if (initialCapacity > 0x20000000) {
            throw new IllegalArgumentException("Initial capacity " + initialCapacity + " exceeds maximum capacity of " + 0x20000000);
        }
        this.allocate(initialCapacity);
    }

    public BytesToBytesMap(TaskMemoryManager memoryManager, int initialCapacity) {
        this(memoryManager, initialCapacity, 0.7, false);
    }

    public BytesToBytesMap(TaskMemoryManager memoryManager, int initialCapacity, boolean enablePerfMetrics) {
        this(memoryManager, initialCapacity, 0.7, enablePerfMetrics);
    }

    public int size() {
        return this.size;
    }

    public Iterator<Location> iterator() {
        return new BytesToBytesMapIterator(this.size, this.dataPages.iterator(), this.loc);
    }

    public Location lookup(Object keyBaseObject, long keyBaseOffset, int keyRowLengthBytes) {
        if (this.enablePerfMetrics) {
            ++this.numKeyLookups;
        }
        int hashcode = HASHER.hashUnsafeWords(keyBaseObject, keyBaseOffset, keyRowLengthBytes);
        int pos = hashcode & this.mask;
        int step = 1;
        while (true) {
            if (this.enablePerfMetrics) {
                ++this.numProbes;
            }
            if (!this.bitset.isSet(pos)) {
                return this.loc.with(pos, hashcode, false);
            }
            long stored = this.longArray.get(pos * 2 + 1);
            if ((int)stored == hashcode) {
                this.loc.with(pos, hashcode, true);
                if (this.loc.getKeyLength() == keyRowLengthBytes) {
                    long storedKeyBaseOffset;
                    MemoryLocation keyAddress = this.loc.getKeyAddress();
                    Object storedKeyBaseObject = keyAddress.getBaseObject();
                    boolean areEqual = ByteArrayMethods.wordAlignedArrayEquals(keyBaseObject, keyBaseOffset, storedKeyBaseObject, storedKeyBaseOffset = keyAddress.getBaseOffset(), keyRowLengthBytes);
                    if (areEqual) {
                        return this.loc;
                    }
                    if (this.enablePerfMetrics) {
                        ++this.numHashCollisions;
                    }
                }
            }
            pos = pos + step & this.mask;
            ++step;
        }
    }

    private void allocate(int capacity) {
        assert (capacity >= 0);
        capacity = Math.max((int)Math.min(0x20000000L, BytesToBytesMap.nextPowerOf2(capacity)), 64);
        assert (capacity <= 0x20000000);
        this.longArray = new LongArray(this.memoryManager.allocate((long)capacity * 8L * 2L));
        this.bitset = new BitSet(MemoryBlock.fromLongArray(new long[capacity / 64]));
        this.growthThreshold = (int)((double)capacity * this.loadFactor);
        this.mask = capacity - 1;
    }

    public void free() {
        if (this.longArray != null) {
            this.memoryManager.free(this.longArray.memoryBlock());
            this.longArray = null;
        }
        if (this.bitset != null) {
            this.bitset = null;
        }
        Iterator<MemoryBlock> dataPagesIterator = this.dataPages.iterator();
        while (dataPagesIterator.hasNext()) {
            this.memoryManager.freePage(dataPagesIterator.next());
            dataPagesIterator.remove();
        }
        assert (this.dataPages.isEmpty());
    }

    public long getTotalMemoryConsumption() {
        return (long)this.dataPages.size() * 0x4000000L + this.bitset.memoryBlock().size() + this.longArray.memoryBlock().size();
    }

    public long getTimeSpentResizingNs() {
        if (!this.enablePerfMetrics) {
            throw new IllegalStateException();
        }
        return this.timeSpentResizingNs;
    }

    public double getAverageProbesPerLookup() {
        if (!this.enablePerfMetrics) {
            throw new IllegalStateException();
        }
        return 1.0 * (double)this.numProbes / (double)this.numKeyLookups;
    }

    public long getNumHashCollisions() {
        if (!this.enablePerfMetrics) {
            throw new IllegalStateException();
        }
        return this.numHashCollisions;
    }

    @VisibleForTesting
    int getNumDataPages() {
        return this.dataPages.size();
    }

    @VisibleForTesting
    void growAndRehash() {
        long resizeStartTime = -1L;
        if (this.enablePerfMetrics) {
            resizeStartTime = System.nanoTime();
        }
        LongArray oldLongArray = this.longArray;
        BitSet oldBitSet = this.bitset;
        int oldCapacity = (int)oldBitSet.capacity();
        this.allocate(Math.min(growthStrategy.nextCapacity(oldCapacity), 0x20000000));
        int pos = oldBitSet.nextSetBit(0);
        while (pos >= 0) {
            long keyPointer = oldLongArray.get(pos * 2);
            int hashcode = (int)oldLongArray.get(pos * 2 + 1);
            int newPos = hashcode & this.mask;
            int step = 1;
            boolean keepGoing = true;
            while (keepGoing) {
                if (!this.bitset.isSet(newPos)) {
                    this.bitset.set(newPos);
                    this.longArray.set(newPos * 2, keyPointer);
                    this.longArray.set(newPos * 2 + 1, hashcode);
                    keepGoing = false;
                    continue;
                }
                newPos = newPos + step & this.mask;
                ++step;
            }
            pos = oldBitSet.nextSetBit(pos + 1);
        }
        this.memoryManager.free(oldLongArray.memoryBlock());
        if (this.enablePerfMetrics) {
            this.timeSpentResizingNs += System.nanoTime() - resizeStartTime;
        }
    }

    private static long nextPowerOf2(long num) {
        long highBit = Long.highestOneBit(num);
        return highBit == num ? num : highBit << 1;
    }

    public final class Location {
        private int pos;
        private boolean isDefined;
        private int keyHashcode;
        private final MemoryLocation keyMemoryLocation = new MemoryLocation();
        private final MemoryLocation valueMemoryLocation = new MemoryLocation();
        private int keyLength;
        private int valueLength;

        private void updateAddressesAndSizes(long fullKeyAddress) {
            this.updateAddressesAndSizes(BytesToBytesMap.this.memoryManager.getPage(fullKeyAddress), BytesToBytesMap.this.memoryManager.getOffsetInPage(fullKeyAddress));
        }

        private void updateAddressesAndSizes(Object page, long keyOffsetInPage) {
            long position = keyOffsetInPage;
            this.keyLength = (int)PlatformDependent.UNSAFE.getLong(page, position);
            this.keyMemoryLocation.setObjAndOffset(page, position += 8L);
            this.valueLength = (int)PlatformDependent.UNSAFE.getLong(page, position += (long)this.keyLength);
            this.valueMemoryLocation.setObjAndOffset(page, position += 8L);
        }

        Location with(int pos, int keyHashcode, boolean isDefined) {
            this.pos = pos;
            this.isDefined = isDefined;
            this.keyHashcode = keyHashcode;
            if (isDefined) {
                long fullKeyAddress = BytesToBytesMap.this.longArray.get(pos * 2);
                this.updateAddressesAndSizes(fullKeyAddress);
            }
            return this;
        }

        Location with(Object page, long keyOffsetInPage) {
            this.isDefined = true;
            this.updateAddressesAndSizes(page, keyOffsetInPage);
            return this;
        }

        public boolean isDefined() {
            return this.isDefined;
        }

        public MemoryLocation getKeyAddress() {
            assert (this.isDefined);
            return this.keyMemoryLocation;
        }

        public int getKeyLength() {
            assert (this.isDefined);
            return this.keyLength;
        }

        public MemoryLocation getValueAddress() {
            assert (this.isDefined);
            return this.valueMemoryLocation;
        }

        public int getValueLength() {
            assert (this.isDefined);
            return this.valueLength;
        }

        public void putNewKey(Object keyBaseObject, long keyBaseOffset, int keyLengthBytes, Object valueBaseObject, long valueBaseOffset, int valueLengthBytes) {
            Object pageBaseObject;
            assert (!this.isDefined) : "Can only set value once for a key";
            assert (keyLengthBytes % 8 == 0);
            assert (valueLengthBytes % 8 == 0);
            if (BytesToBytesMap.this.size == 0x20000000) {
                throw new IllegalStateException("BytesToBytesMap has reached maximum capacity");
            }
            long requiredSize = 8 + keyLengthBytes + 8 + valueLengthBytes;
            assert (requiredSize <= 0x3FFFFF8L);
            BytesToBytesMap.this.size++;
            BytesToBytesMap.this.bitset.set(this.pos);
            if (BytesToBytesMap.this.currentDataPage == null || 0x3FFFFF8L - BytesToBytesMap.this.pageCursor < requiredSize) {
                if (BytesToBytesMap.this.currentDataPage != null) {
                    pageBaseObject = BytesToBytesMap.this.currentDataPage.getBaseObject();
                    long lengthOffsetInPage = BytesToBytesMap.this.currentDataPage.getBaseOffset() + BytesToBytesMap.this.pageCursor;
                    PlatformDependent.UNSAFE.putLong(pageBaseObject, lengthOffsetInPage, -1L);
                }
                MemoryBlock newPage = BytesToBytesMap.this.memoryManager.allocatePage(0x4000000L);
                BytesToBytesMap.this.dataPages.add(newPage);
                BytesToBytesMap.this.pageCursor = 0L;
                BytesToBytesMap.this.currentDataPage = newPage;
            }
            pageBaseObject = BytesToBytesMap.this.currentDataPage.getBaseObject();
            long pageBaseOffset = BytesToBytesMap.this.currentDataPage.getBaseOffset();
            long keySizeOffsetInPage = pageBaseOffset + BytesToBytesMap.this.pageCursor;
            BytesToBytesMap.this.pageCursor += 8L;
            long keyDataOffsetInPage = pageBaseOffset + BytesToBytesMap.this.pageCursor;
            BytesToBytesMap.this.pageCursor += keyLengthBytes;
            long valueSizeOffsetInPage = pageBaseOffset + BytesToBytesMap.this.pageCursor;
            BytesToBytesMap.this.pageCursor += 8L;
            long valueDataOffsetInPage = pageBaseOffset + BytesToBytesMap.this.pageCursor;
            BytesToBytesMap.this.pageCursor += valueLengthBytes;
            PlatformDependent.UNSAFE.putLong(pageBaseObject, keySizeOffsetInPage, keyLengthBytes);
            PlatformDependent.copyMemory(keyBaseObject, keyBaseOffset, pageBaseObject, keyDataOffsetInPage, keyLengthBytes);
            PlatformDependent.UNSAFE.putLong(pageBaseObject, valueSizeOffsetInPage, valueLengthBytes);
            PlatformDependent.copyMemory(valueBaseObject, valueBaseOffset, pageBaseObject, valueDataOffsetInPage, valueLengthBytes);
            long storedKeyAddress = BytesToBytesMap.this.memoryManager.encodePageNumberAndOffset(BytesToBytesMap.this.currentDataPage, keySizeOffsetInPage);
            BytesToBytesMap.this.longArray.set(this.pos * 2, storedKeyAddress);
            BytesToBytesMap.this.longArray.set(this.pos * 2 + 1, this.keyHashcode);
            this.updateAddressesAndSizes(storedKeyAddress);
            this.isDefined = true;
            if (BytesToBytesMap.this.size > BytesToBytesMap.this.growthThreshold && BytesToBytesMap.this.longArray.size() < 0x20000000L) {
                BytesToBytesMap.this.growAndRehash();
            }
        }
    }

    private static final class BytesToBytesMapIterator
    implements Iterator<Location> {
        private final int numRecords;
        private final Iterator<MemoryBlock> dataPagesIterator;
        private final Location loc;
        private int currentRecordNumber = 0;
        private Object pageBaseObject;
        private long offsetInPage;

        BytesToBytesMapIterator(int numRecords, Iterator<MemoryBlock> dataPagesIterator, Location loc) {
            this.numRecords = numRecords;
            this.dataPagesIterator = dataPagesIterator;
            this.loc = loc;
            if (dataPagesIterator.hasNext()) {
                this.advanceToNextPage();
            }
        }

        private void advanceToNextPage() {
            MemoryBlock currentPage = this.dataPagesIterator.next();
            this.pageBaseObject = currentPage.getBaseObject();
            this.offsetInPage = currentPage.getBaseOffset();
        }

        @Override
        public boolean hasNext() {
            return this.currentRecordNumber != this.numRecords;
        }

        @Override
        public Location next() {
            int keyLength = (int)PlatformDependent.UNSAFE.getLong(this.pageBaseObject, this.offsetInPage);
            if (keyLength == -1) {
                this.advanceToNextPage();
                keyLength = (int)PlatformDependent.UNSAFE.getLong(this.pageBaseObject, this.offsetInPage);
            }
            this.loc.with(this.pageBaseObject, this.offsetInPage);
            this.offsetInPage += (long)(16 + keyLength + this.loc.getValueLength());
            ++this.currentRecordNumber;
            return this.loc;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

