/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.DiskValidator;
import org.apache.hadoop.util.DiskValidatorFactory;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.exceptions.YarnRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryCollection {
    private static final Logger LOG = LoggerFactory.getLogger(DirectoryCollection.class);
    private final Configuration conf = new YarnConfiguration();
    private final DiskValidator diskValidator;
    private List<String> localDirs;
    private List<String> errorDirs;
    private List<String> fullDirs;
    private Map<String, DiskErrorInformation> directoryErrorInfo;
    private final ReentrantReadWriteLock.ReadLock readLock;
    private final ReentrantReadWriteLock.WriteLock writeLock;
    private int numFailures;
    private float diskUtilizationPercentageCutoffHigh;
    private float diskUtilizationPercentageCutoffLow;
    private long diskUtilizationSpaceCutoff;
    private int goodDirsDiskUtilizationPercentage;
    private Set<DirsChangeListener> dirsChangeListeners;

    static List<String> concat(List<String> l1, List<String> l2) {
        ArrayList<String> ret = new ArrayList<String>(l1.size() + l2.size());
        ret.addAll(l1);
        ret.addAll(l2);
        return ret;
    }

    public DirectoryCollection(String[] dirs) {
        this(dirs, 100.0f, 100.0f, 0L);
    }

    public DirectoryCollection(String[] dirs, float utilizationPercentageCutOff) {
        this(dirs, utilizationPercentageCutOff, utilizationPercentageCutOff, 0L);
    }

    public DirectoryCollection(String[] dirs, long utilizationSpaceCutOff) {
        this(dirs, 100.0f, 100.0f, utilizationSpaceCutOff);
    }

    public DirectoryCollection(String[] dirs, float utilizationPercentageCutOffHigh, float utilizationPercentageCutOffLow, long utilizationSpaceCutOff) {
        try {
            this.diskValidator = DiskValidatorFactory.getInstance((String)this.conf.get("yarn.nodemanager.disk-validator", "basic"));
            LOG.info("Disk Validator: yarn.nodemanager.disk-validator is loaded.");
        }
        catch (Exception e) {
            throw new YarnRuntimeException((Throwable)e);
        }
        this.localDirs = new CopyOnWriteArrayList<String>(dirs);
        this.errorDirs = new CopyOnWriteArrayList<String>();
        this.fullDirs = new CopyOnWriteArrayList<String>();
        this.directoryErrorInfo = new ConcurrentHashMap<String, DiskErrorInformation>();
        ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
        this.readLock = lock.readLock();
        this.writeLock = lock.writeLock();
        this.diskUtilizationPercentageCutoffHigh = Math.max(0.0f, Math.min(100.0f, utilizationPercentageCutOffHigh));
        this.diskUtilizationPercentageCutoffLow = Math.max(0.0f, Math.min(this.diskUtilizationPercentageCutoffHigh, utilizationPercentageCutOffLow));
        this.diskUtilizationSpaceCutoff = utilizationSpaceCutOff < 0L ? 0L : utilizationSpaceCutOff;
        this.dirsChangeListeners = Collections.newSetFromMap(new ConcurrentHashMap());
    }

    void registerDirsChangeListener(DirsChangeListener listener) {
        if (this.dirsChangeListeners.add(listener)) {
            listener.onDirsChanged();
        }
    }

    void deregisterDirsChangeListener(DirsChangeListener listener) {
        this.dirsChangeListeners.remove(listener);
    }

    List<String> getGoodDirs() {
        this.readLock.lock();
        try {
            List<String> list = Collections.unmodifiableList(this.localDirs);
            return list;
        }
        finally {
            this.readLock.unlock();
        }
    }

    List<String> getFailedDirs() {
        this.readLock.lock();
        try {
            List<String> list = Collections.unmodifiableList(DirectoryCollection.concat(this.errorDirs, this.fullDirs));
            return list;
        }
        finally {
            this.readLock.unlock();
        }
    }

    List<String> getFullDirs() {
        this.readLock.lock();
        try {
            List<String> list = Collections.unmodifiableList(this.fullDirs);
            return list;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    List<String> getErroredDirs() {
        this.readLock.lock();
        try {
            List<String> list = Collections.unmodifiableList(this.errorDirs);
            return list;
        }
        finally {
            this.readLock.unlock();
        }
    }

    int getNumFailures() {
        this.readLock.lock();
        try {
            int n = this.numFailures;
            return n;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    DiskErrorInformation getDirectoryErrorInfo(String dirName) {
        this.readLock.lock();
        try {
            DiskErrorInformation diskErrorInformation = this.directoryErrorInfo.get(dirName);
            return diskErrorInformation;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @InterfaceStability.Evolving
    boolean isDiskUnHealthy(String dirName) {
        this.readLock.lock();
        try {
            boolean bl = this.directoryErrorInfo.containsKey(dirName);
            return bl;
        }
        finally {
            this.readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean createNonExistentDirs(FileContext localFs, FsPermission perm) {
        boolean failed = false;
        ArrayList<String> localDirectories = null;
        this.readLock.lock();
        try {
            localDirectories = new ArrayList<String>(this.localDirs);
        }
        finally {
            this.readLock.unlock();
        }
        for (String dir : localDirectories) {
            try {
                this.createDir(localFs, new Path(dir), perm);
            }
            catch (IOException e) {
                LOG.warn("Unable to create directory " + dir + " error " + e.getMessage() + ", removing from the list of valid directories.");
                this.writeLock.lock();
                try {
                    this.localDirs.remove(dir);
                    this.errorDirs.add(dir);
                    this.directoryErrorInfo.put(dir, new DiskErrorInformation(DiskErrorCause.OTHER, "Cannot create directory : " + dir + ", error " + e.getMessage()));
                    ++this.numFailures;
                }
                finally {
                    this.writeLock.unlock();
                }
                failed = true;
            }
        }
        return !failed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean checkDirs() {
        boolean setChanged = false;
        HashSet<String> preCheckGoodDirs = null;
        HashSet<String> preCheckFullDirs = null;
        HashSet<String> preCheckOtherErrorDirs = null;
        List<String> failedDirs = null;
        List<String> allLocalDirs = null;
        this.readLock.lock();
        try {
            preCheckGoodDirs = new HashSet<String>(this.localDirs);
            preCheckFullDirs = new HashSet<String>(this.fullDirs);
            preCheckOtherErrorDirs = new HashSet<String>(this.errorDirs);
            failedDirs = DirectoryCollection.concat(this.errorDirs, this.fullDirs);
            allLocalDirs = DirectoryCollection.concat(this.localDirs, failedDirs);
        }
        finally {
            this.readLock.unlock();
        }
        Map<String, DiskErrorInformation> dirsFailedCheck = this.testDirs(allLocalDirs, preCheckGoodDirs);
        this.writeLock.lock();
        try {
            this.localDirs.clear();
            this.errorDirs.clear();
            this.fullDirs.clear();
            this.directoryErrorInfo.clear();
            for (Map.Entry<String, DiskErrorInformation> entry : dirsFailedCheck.entrySet()) {
                String dir = entry.getKey();
                DiskErrorInformation errorInformation = entry.getValue();
                switch (entry.getValue().cause) {
                    case DISK_FULL: {
                        this.fullDirs.add(entry.getKey());
                        break;
                    }
                    case OTHER: {
                        this.errorDirs.add(entry.getKey());
                        break;
                    }
                    default: {
                        LOG.warn((Object)((Object)entry.getValue().cause) + " is unknown for disk error.");
                    }
                }
                this.directoryErrorInfo.put(entry.getKey(), errorInformation);
                if (!preCheckGoodDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error, " + errorInformation.message + ", removing from list of valid directories");
                setChanged = true;
                ++this.numFailures;
            }
            for (String dir : allLocalDirs) {
                if (dirsFailedCheck.containsKey(dir)) continue;
                this.localDirs.add(dir);
                if (!preCheckFullDirs.contains(dir) && !preCheckOtherErrorDirs.contains(dir)) continue;
                setChanged = true;
                LOG.info("Directory " + dir + " passed disk check, adding to list of valid directories.");
            }
            HashSet<String> postCheckFullDirs = new HashSet<String>(this.fullDirs);
            HashSet<String> postCheckOtherDirs = new HashSet<String>(this.errorDirs);
            for (String dir : preCheckFullDirs) {
                if (!postCheckOtherDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error " + dirsFailedCheck.get((Object)dir).message);
            }
            for (String dir : preCheckOtherErrorDirs) {
                if (!postCheckFullDirs.contains(dir)) continue;
                LOG.warn("Directory " + dir + " error " + dirsFailedCheck.get((Object)dir).message);
            }
            this.setGoodDirsDiskUtilizationPercentage();
            if (setChanged) {
                for (DirsChangeListener listener : this.dirsChangeListeners) {
                    listener.onDirsChanged();
                }
            }
            boolean bl = setChanged;
            return bl;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    Map<String, DiskErrorInformation> testDirs(List<String> dirs, Set<String> goodDirs) {
        HashMap<String, DiskErrorInformation> ret = new HashMap<String, DiskErrorInformation>();
        for (String dir : dirs) {
            try {
                String msg;
                float diskUtilizationPercentageCutoff;
                File testDir = new File(dir);
                this.diskValidator.checkStatus(testDir);
                float f = diskUtilizationPercentageCutoff = goodDirs.contains(dir) ? this.diskUtilizationPercentageCutoffHigh : this.diskUtilizationPercentageCutoffLow;
                if (this.isDiskUsageOverPercentageLimit(testDir, diskUtilizationPercentageCutoff)) {
                    msg = "used space above threshold of " + diskUtilizationPercentageCutoff + "%";
                    ret.put(dir, new DiskErrorInformation(DiskErrorCause.DISK_FULL, msg));
                    continue;
                }
                if (this.isDiskFreeSpaceUnderLimit(testDir)) {
                    msg = "free space below limit of " + this.diskUtilizationSpaceCutoff + "MB";
                    ret.put(dir, new DiskErrorInformation(DiskErrorCause.DISK_FULL, msg));
                    continue;
                }
                this.verifyDirUsingMkdir(testDir);
            }
            catch (IOException ie) {
                ret.put(dir, new DiskErrorInformation(DiskErrorCause.OTHER, ie.getMessage()));
            }
        }
        return ret;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyDirUsingMkdir(File dir) throws IOException {
        String randomDirName = RandomStringUtils.randomAlphanumeric((int)5);
        File target = new File(dir, randomDirName);
        int i = 0;
        while (target.exists()) {
            randomDirName = RandomStringUtils.randomAlphanumeric((int)5) + i;
            target = new File(dir, randomDirName);
            ++i;
        }
        try {
            this.diskValidator.checkStatus(target);
        }
        finally {
            FileUtils.deleteQuietly((File)target);
        }
    }

    private boolean isDiskUsageOverPercentageLimit(File dir, float diskUtilizationPercentageCutoff) {
        float freePercentage = 100.0f * ((float)dir.getUsableSpace() / (float)dir.getTotalSpace());
        float usedPercentage = 100.0f - freePercentage;
        return usedPercentage > diskUtilizationPercentageCutoff || usedPercentage >= 100.0f;
    }

    private boolean isDiskFreeSpaceUnderLimit(File dir) {
        long freeSpace = dir.getUsableSpace() / 0x100000L;
        return freeSpace < this.diskUtilizationSpaceCutoff;
    }

    private void createDir(FileContext localFs, Path dir, FsPermission perm) throws IOException {
        block5: {
            if (dir == null) {
                return;
            }
            try {
                localFs.getFileStatus(dir);
            }
            catch (FileNotFoundException e) {
                this.createDir(localFs, dir.getParent(), perm);
                try {
                    localFs.mkdir(dir, perm, false);
                }
                catch (FileAlreadyExistsException fileAlreadyExistsException) {
                    // empty catch block
                }
                if (perm.equals((Object)perm.applyUMask(localFs.getUMask()))) break block5;
                localFs.setPermission(dir, perm);
            }
        }
    }

    @VisibleForTesting
    float getDiskUtilizationPercentageCutoffHigh() {
        return this.diskUtilizationPercentageCutoffHigh;
    }

    @VisibleForTesting
    float getDiskUtilizationPercentageCutoffLow() {
        return this.diskUtilizationPercentageCutoffLow;
    }

    public void setDiskUtilizationPercentageCutoff(float utilizationPercentageCutOffHigh, float utilizationPercentageCutOffLow) {
        this.diskUtilizationPercentageCutoffHigh = Math.max(0.0f, Math.min(100.0f, utilizationPercentageCutOffHigh));
        this.diskUtilizationPercentageCutoffLow = Math.max(0.0f, Math.min(this.diskUtilizationPercentageCutoffHigh, utilizationPercentageCutOffLow));
    }

    public long getDiskUtilizationSpaceCutoff() {
        return this.diskUtilizationSpaceCutoff;
    }

    public void setDiskUtilizationSpaceCutoff(long diskUtilizationSpaceCutoff) {
        this.diskUtilizationSpaceCutoff = diskUtilizationSpaceCutoff = diskUtilizationSpaceCutoff < 0L ? 0L : diskUtilizationSpaceCutoff;
    }

    private void setGoodDirsDiskUtilizationPercentage() {
        long totalSpace = 0L;
        long usableSpace = 0L;
        for (String dir : this.localDirs) {
            File f = new File(dir);
            if (!f.isDirectory()) continue;
            totalSpace += f.getTotalSpace();
            usableSpace += f.getUsableSpace();
        }
        if (totalSpace != 0L) {
            long tmp = (totalSpace - usableSpace) * 100L / totalSpace;
            if (Integer.MIN_VALUE < tmp && Integer.MAX_VALUE > tmp) {
                this.goodDirsDiskUtilizationPercentage = (int)tmp;
            }
        } else {
            this.goodDirsDiskUtilizationPercentage = 0;
        }
    }

    public int getGoodDirsDiskUtilizationPercentage() {
        return this.goodDirsDiskUtilizationPercentage;
    }

    public static interface DirsChangeListener {
        public void onDirsChanged();
    }

    static class DiskErrorInformation {
        DiskErrorCause cause;
        String message;

        DiskErrorInformation(DiskErrorCause cause, String message) {
            this.cause = cause;
            this.message = message;
        }
    }

    public static enum DiskErrorCause {
        DISK_FULL,
        OTHER;

    }
}

