package org.beast.user.config;

import com.google.common.collect.Lists;
import org.beast.security.core.*;
import org.beast.user.core.AuthenticationKey;
import org.beast.user.core.Identity;
import org.beast.user.core.IdentityType;

import java.util.List;
import java.util.Objects;

public class SNSIdentityUtils {


    public static String identifierOfSNSToken(SNSToken token) {
        return switch (token.getType()) {
            case WECHAT_WEB -> ((WechatWebSNSToken) token).getOpenid();
            case WECHAT_OFFIACCOUNT -> ((WechatOffiaccountSNSToken) token).getOpenid();
            case WECHAT_WEAPP -> ((WechatWeappSNSToken) token).getOpenid();
            case WECHAT_UNION -> ((WechatUnionSNSToken) token).getUnionId();
            case ALIPAY_ALIAPP -> ((AlipayAliappSNSToken) token).getUserId();
            case BYTEDANCE_BYTEAPP ->  ((BytedanceByteappSNSToken) token).getOpenid();
            default -> {
                throw new IllegalArgumentException("No matching type [" + token.getType() + "]");
            }
        };
    }

    public static SNSType typeOfIdentityType(IdentityType identityType) {

        return switch (identityType) {
            case WECHAT_WEB -> SNSType.WECHAT_WEB;
            case WECHAT_OFFIACCOUNT -> SNSType.WECHAT_OFFIACCOUNT;
            case WECHAT_WEAPP -> SNSType.WECHAT_WEAPP;
            case WECHAT_UNION -> SNSType.WECHAT_UNION;
            case ALIPAY_ALIAPP -> SNSType.ALIPAY_ALIAPP;
            case BYTEDANCE_BYTEAPP -> SNSType.BYTEDANCE_BYTEAPP;
            default -> {
                throw new IllegalArgumentException("No matching type [" + identityType + "]");
            }
        };
    }
    public static IdentityType identityTypeOfSNSType(SNSType type) {
        return switch (type) {
            case WECHAT_WEB -> IdentityType.WECHAT_WEB;
            case WECHAT_OFFIACCOUNT -> IdentityType.WECHAT_OFFIACCOUNT;
            case WECHAT_WEAPP -> IdentityType.WECHAT_WEAPP;
            case WECHAT_UNION -> IdentityType.WECHAT_UNION;
            case ALIPAY_ALIAPP -> IdentityType.ALIPAY_ALIAPP;
            case BYTEDANCE_BYTEAPP -> IdentityType.BYTEDANCE_BYTEAPP;
            default -> {
                throw new IllegalArgumentException("No matching type [" + type + "]");
            }
        };
    }

    public static Identity identityOfAuthenticationKeyAndToken(
            AuthenticationKey authenticationKey,
            SNSToken snsToken
    ) {
        var identityType = SNSIdentityUtils.identityTypeOfSNSType(snsToken.getType());
        if (!authenticationKey.isIdentityType(identityType)) {
            throw new IllegalStateException(
                    String.format("authenticationKey [%s] unable resolve token type: [%s] ",
                            authenticationKey,
                            snsToken.getType()
                    )
            );
        }
        return new Identity(identityType, authenticationKey.getGroup(), SNSIdentityUtils.identifierOfSNSToken(snsToken));
    }

    public static boolean support(IdentityType identityType) {
        for (SNSType value : SNSType.values()) {
            if (Objects.equals(value.name(), identityType.name())) {
                return true;
            }
        }
        return false;
    }

    public static List<Identity> identityOfUserAppKeySNSIdentity(UserApp app, AuthenticationKey authenticationKey, SNSToken token) {
        var type = token.getType();
        List<Identity> identities = Lists.newArrayList(
            identityOfAuthenticationKeyAndToken(authenticationKey, token)
        );
        switch (type) {
            case WECHAT_WEAPP -> {
                WechatWeappSNSToken weappSNSToken = (WechatWeappSNSToken) token;
                var unionId = weappSNSToken.getUnionId();
                if (Objects.nonNull(unionId)) {
                    var unionKey = app.lookupUnionKey(authenticationKey);
                    var unionIdentity = Identity.valueOf(
                            AuthenticationKey.valueOf(IdentityType.WECHAT_UNION, unionKey),
                            unionId
                    );
                    identities.add(unionIdentity);
                }
            }
            case WECHAT_OFFIACCOUNT -> {
                WechatOffiaccountSNSToken offiaccountSNSToken = (WechatOffiaccountSNSToken) token;
                var unionId = offiaccountSNSToken.getUnionId();
                if (Objects.nonNull(unionId)) {
                    var unionKey = app.lookupUnionKey(authenticationKey);
                    var unionIdentity = Identity.valueOf(
                            AuthenticationKey.valueOf(IdentityType.WECHAT_UNION, unionKey),
                            unionId
                    );
                    identities.add(unionIdentity);
                }
            }

//            case WECHAT_WEB ->
        }
        return identities;
    }
}
