package org.beast.user.config;

import com.google.common.collect.Lists;
import org.beast.user.core.AuthenticationKey;
import org.beast.user.core.IdentityType;
import lombok.Getter;
import lombok.Setter;
import org.beast.user.core.LoginType;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


public class AppEntrances {

    private Map<String, Config> entrances;

    public AppEntrances(Map<String, Config> entrances) {
        this.entrances = entrances;
    }

    public boolean support(String key) {
        return this.entrances.containsKey(key);
    }


    private Config getEntranceConfig(String entrance) {
        Config config = entrances.get(entrance);
        if (config == null) {
            throw new IllegalStateException("entrance: "+ entrance + " entrance-config invalid ");
        }
        return config;
    }

    public boolean enableEntrance(String entrance) {
        return entrances.containsKey(entrance);
    }

    public AuthenticationKey lookup(String entrance, IdentityType type) {
        AppEntrances.Config config = getEntranceConfig(entrance);
        for (AuthenticationKey authentication : config.getAuthentications()) {
            if (authentication.getIdentityType() == type) {
                return authentication;
            }
        }
        throw new IllegalStateException("Unsupported entrance:'" + entrance + "' type: '" + type + "'");
    }

    public boolean enableAuthentication(String entrance, IdentityType type) {
        AppEntrances.Config config = getEntranceConfig(entrance);
        for (AuthenticationKey authentication : config.authentications) {
            if (authentication.getIdentityType() == type) {
                return true;
            }
        }
        return false;
    }
    public boolean isEnable(String entrance, EntranceFeature feature) {
        var config = getEntranceConfig(entrance);
        var features = config.getFeatures();
        Boolean value = features.get(feature.name());
        if (value == null) {
            return feature.enabledByDefault();
        }
        return value;
    }

    public List<LoginType> getLoginTypes(String entrance) {
        var config = getEntranceConfig(entrance);
        return config.getLoginTypes();
    }
    public List<IdentityType> types(String entrance) {
        var config = getEntranceConfig(entrance);
        return config.getAuthentications().stream().map(AuthenticationKey::getIdentityType).collect(Collectors.toList());
    }



    @Getter @Setter
    public static class Config {

        //登录类型：密码登录，手机号登录，SNSLogin
        private List<LoginType> loginTypes;

        private Map<String, String> metadata = new HashMap<>();

        private Map<String, Boolean> features = new HashMap<>();

        private List<AuthenticationKey> authentications = Lists.newArrayList();

        public boolean enable = true;
    }
}
