package org.beast.user.config;

import com.google.common.collect.Lists;
import org.beast.user.core.*;
import org.beast.user.core.http.CookieControl;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.Nullable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 应用
 */
@Slf4j
@Setter
public class UserApp {


    @Getter
    private String id;

    /**
     * 名称
     */
    @Getter
    private String name;


    /**
     * LOGO 42x48
     */
    @Getter
    private String logo;

    /**
     * redirectUrl 使用
     */
    //private String baseUrl;
    /**
     * baseUrl: http://webpassport.office.zhangrc.site/
     * baseUrl: http://h5.pingtech.net/login/
     * baseUrl: http://webpassport.pingtech.net/
     */
    @Getter
    private Map<String, String> metadata;

    @Getter
    private Map<String, Boolean> features = new HashMap<>();

    @Getter
    private CookieControl cookieControl;


    private EntranceConfig entrances;


    private AuthenticationConfig authentications;

    public UserApp(AppProperties properties) {
        this.id = properties.getId();
        this.name = properties.getName();
        this.logo = properties.getLogo();
        this.metadata = properties.getMetadata();
        this.features = properties.getFeatures();
        this.cookieControl = properties.getCookieControl();
        this.entrances = new EntranceConfig(properties.getEntrances());
        this.authentications = new AuthenticationConfig(
                properties.getAuthentications().stream().collect(Collectors.toMap(
                        AuthenticationConfig.Config::getKey,
                        item -> item
                ))
        );
    }

    public String id() {
        return this.id;
    }
    public boolean enableByEntrance(String entrance) {
        if (entrance == null) {
            entrance = id;
        }
        return entrances.enableEntrance(entrance);
    }
    public boolean enableByEntrance(String entrance, EntranceFeature feature) {
        if (entrance == null) {
            entrance = id;
        }
        return entrances.isEnable(entrance, feature);
    }

    public boolean enableByAuthenticateFeature(AuthenticationKey key, AuthenticationFeature feature) {
        return this.authentications.isEnable(key, feature);
    }

    public boolean enableByAuthenticateFeature(String entrance, IdentityType type, AuthenticationFeature feature) {
        if (entrance == null) {
            entrance = id;
        }
        AuthenticationKey key = entrances.lookup(entrance, type);
        return enableByAuthenticateFeature(key, feature);
    }

    public boolean enableAuthenticationType(String entrance, IdentityType identityType) {
        if (entrance == null) {
            entrance = id;
        }
        return this.entrances.enableAuthentication(entrance, identityType);
    }
    public AuthenticationKey authenticationKey(String entrance, IdentityType identityType) {
        if (entrance == null) {
            entrance = id;
        }
        return this.entrances.lookup(entrance, identityType);
    }

    public String getSNSMetadataValue(String entrance, SNSType type, String field) {
        if (entrance == null) {
            entrance = id;
        }
        IdentityType identityType = type.toIdentityType();
        AuthenticationKey key = entrances.lookup(entrance, identityType);
        AuthenticationConfig.Config config = this.authentications.getAuthenticationConfig(key);
        if (config == null) {
            throw new IllegalStateException("authentication key: " + key + " config unsupported, supports [{" +this.authentications.keys()+ "}]");
        }
        Map<String, String> metadata = config.getMetadata();
        return metadata.get(field);
    }
    public String lookupSNSApp(@Nullable String entrance, SNSType type) {
        return getSNSMetadataValue(entrance, type, "app");
    }

    public List<LoginType> getLoginTypes(@Nullable String entrance) {
        if (entrance == null) {
            entrance = id;
        }
        return entrances.getLoginTypes(entrance);
    }
    public List<SNSType> getSNSTypes(@Nullable String entrance) {
        if (entrance == null) {
            entrance = id;
        }
        List<IdentityType> types = entrances.types(entrance);

        List<SNSType> output = Lists.newArrayList();
        for (IdentityType type : types) {
            try {
                output.add(SNSType.valueOfIdentityType(type));
            } catch (IllegalArgumentException ignore) {

            }
        }
        return output;
    }

    public boolean own(Apped apped) {
        return this.id.equals(apped.appId());
    }
}
