package org.beast.security.web.resolver;

import lombok.extern.slf4j.Slf4j;
import org.beast.security.core.UserToken;
import org.beast.security.core.codec.UserTokenCodec;
import org.beast.security.core.exception.TokenException;
import org.beast.web.util.WebUtils;
import org.springframework.core.MethodParameter;
import org.springframework.util.ObjectUtils;
import org.springframework.web.context.request.NativeWebRequest;

import javax.servlet.http.HttpServletRequest;

@Slf4j
public class UserTokenWebExtractor implements WebExtractor<UserToken>{

    private WebExtractor<String> tokenStringWebExtractor;

    private final UserTokenCodec codec = new UserTokenCodec();

    public UserTokenWebExtractor() {
        super();
        tokenStringWebExtractor = CompositeTokenValueWebExtractor.of(
                new HeaderTokenValueWebExtractor(WebUtils.HEADER_X_U_TOKEN),
                new CookieTokenValueWebExtractor(WebUtils.COOKIE_U_TOKEN)
        );
    }
    @Override
    public UserToken extract(MethodParameter parameter, NativeWebRequest webRequest) {
        String tokenString = tokenStringWebExtractor.extract(parameter, webRequest);
        return parseToken(tokenString);
    }

    @Override
    public UserToken extract(HttpServletRequest servletRequest) {
        String tokenString = tokenStringWebExtractor.extract(servletRequest);
        return parseToken(tokenString);
    }

    public UserToken parseToken(String tokenString) {
        if (ObjectUtils.isEmpty(tokenString)) {
            return null;
        }
        UserToken userToken = null;
        try {
            userToken = codec.decode(tokenString);
            userToken.verify();
        } catch (TokenException e) {
            log.warn("UserToken extract exception, token: {}", tokenString, e);
            return null;
        } catch (Exception e) {
            log.warn("UserToken extract failure, token:{}", tokenString, e);
            return null;
        }
        return userToken;
    }
}
