package org.beast.security.web.resolver;


import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.beast.security.core.*;
import org.beast.security.core.annotation.SNSUserTokenValue;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.context.request.NativeWebRequest;

import java.util.Objects;

@Slf4j
public class SNSUserTokenArgumentResolver extends AbstractTokenArgumentResolver<SNSUserTokenValue> {

    private SNSUserTokenWebExtractor snsUserTokenWebExtractor;

    public SNSUserTokenArgumentResolver() {
        super(SNSUserTokenValue.class);
        this.snsUserTokenWebExtractor = new SNSUserTokenWebExtractor();
    }

    @Override
    public boolean isRequired(SNSUserTokenValue annotation) {
        return annotation.required();
    }


    @Override
    protected void handleMissingValue(MethodParameter parameter) throws ServletRequestBindingException {
        throw new AccessDeniedException("SNSUserToken is required");
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return AnnotatedElementUtils.hasAnnotation(parameter.getParameter(), SNSUserTokenValue.class);
//        parameter.hasParameterAnnotation(SNSUserTokenValue)
    }

    @Override
    public SNSUserTokenValue resolveAnnotation(MethodParameter parameter) {
        return AnnotatedElementUtils.findMergedAnnotation(parameter.getParameter(), SNSUserTokenValue.class);
    }

    @Override
    protected Object resolveValue(SNSUserTokenValue annotation, MethodParameter parameter, NativeWebRequest webRequest) throws ServletRequestBindingException {
        SNSUserToken snsUserToken = snsUserTokenWebExtractor.extract(parameter, webRequest);
        if (Objects.isNull(snsUserToken)) {
            return null;
        }
        SNSTokenType[] requiredTypes = annotation.type();
        if (requiredTypes.length > 0) {
            if (!ArrayUtils.contains(requiredTypes, snsUserToken.getType())) {
                return null;
            }
        }
        if (String.class.isAssignableFrom(parameter.getNestedParameterType())) {
            return switch (snsUserToken.getType()) {
                case WECHAT_WEAPP -> ((WechatWeappSNSUserToken)snsUserToken).getOpenid();
                case WECHAT_OFFIACCOUNT -> ((WechatOffiAccountSNSUserToken)snsUserToken).getOpenid();
                case BYTEDANCE_BYTEAPP -> ((BytedanceByteappSNSUserToken)snsUserToken).getOpenid();
                default -> null;
            };
        } else {
            return snsUserToken;
        }
    }
}
