package org.beast.security.web.resolver;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.MethodParameter;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.annotation.MethodArgumentConversionNotSupportedException;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.springframework.web.util.UrlPathHelper;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;

@Slf4j
public abstract class AbstractTokenArgumentResolver<A extends Annotation> implements HandlerMethodArgumentResolver {

    private UrlPathHelper urlPathHelper = new UrlPathHelper();

    private Class<A> annotationClass;



    public AbstractTokenArgumentResolver(Class<A> annotationClass) {
        this.annotationClass = annotationClass;
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return parameter.hasParameterAnnotation(this.annotationClass);
    }

    public abstract boolean isRequired(A annotation);

    public A resolveAnnotation(MethodParameter parameter) {
        return parameter.getParameterAnnotation(this.annotationClass);
    }

    @Nullable
    @Override
    public Object resolveArgument(@NonNull MethodParameter parameter, @Nullable ModelAndViewContainer mavContainer,
                                  @NonNull NativeWebRequest webRequest, @Nullable WebDataBinderFactory binderFactory) throws Exception {
        A annotation = resolveAnnotation(parameter);
        Assert.state(annotation != null, "No " + annotationClass.getSimpleName() + " annotation");
        String name = parameter.getParameterName();
        if (name == null) {
            name = "";
        }
        MethodParameter nestedParameter = parameter.nestedIfOptional();
        Object arg = resolveValue(annotation, nestedParameter, webRequest);
        if (arg == null) {
            if (isRequired(annotation) && !nestedParameter.isOptional()) {
                handleMissingValue(parameter);
            }
            arg = handleNullValue(arg, nestedParameter.getNestedParameterType());
        }
        if (binderFactory != null) {
            WebDataBinder binder = binderFactory.createBinder(webRequest, null, name);
            try {
                arg = binder.convertIfNecessary(arg, parameter.getParameterType(), parameter);
            }
            catch (ConversionNotSupportedException ex) {
                throw new MethodArgumentConversionNotSupportedException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
            catch (TypeMismatchException ex) {
                throw new MethodArgumentTypeMismatchException(arg, ex.getRequiredType(),
                        name, parameter, ex.getCause());
            }
        }
        return arg;
    }

    protected abstract void handleMissingValue(MethodParameter parameter) throws ServletRequestBindingException;


    @Nullable
    private Object handleNullValue(@Nullable Object value, Class<?> paramType) {
        if (value == null) {
            if (Boolean.TYPE.equals(paramType)) {
                return Boolean.FALSE;
            }
        }
        return value;
    }

//    protected abstract Object resolveToken(MethodParameter parameter, String tokenString) throws ServletRequestBindingException;

    protected abstract @Nullable Object resolveValue(A annotation, MethodParameter parameter, NativeWebRequest webRequest) throws ServletRequestBindingException;
//        HttpServletRequest servletRequest = webRequest.getNativeRequest(HttpServletRequest.class);
//        Assert.state(servletRequest != null, "No HttpServletRequest");
//        String tokenString = null;
//        if (headerName != null) {
//            String headerValue = servletRequest.getHeader(this.headerName);
//            if (headerValue != null) {
//                tokenString = this.urlPathHelper.decodeRequestString(servletRequest, headerValue);
//            }
//        }
//        if (tokenString == null && this.cookieName != null){
//            Cookie cookieValue = WebUtils.getCookie(servletRequest, this.cookieName);
//            if (cookieValue != null) {
//                tokenString = this.urlPathHelper.decodeRequestString(servletRequest, cookieValue.getValue());
//            }
//        }
//        if (tokenString == null) {
//            return null;
//        }
//        return resolveToken(parameter, tokenString);
//    }
}
