package org.beast.security.web.resolver;

import lombok.extern.slf4j.Slf4j;
import org.beast.security.core.SNSUserToken;
import org.beast.security.core.codec.SNSUserTokenCodec;
import org.beast.security.core.exception.TokenException;
import org.beast.web.util.WebUtils;
import org.springframework.core.MethodParameter;
import org.springframework.util.ObjectUtils;
import org.springframework.web.context.request.NativeWebRequest;

import javax.servlet.http.HttpServletRequest;

@Slf4j
public class SNSUserTokenWebExtractor implements WebExtractor<SNSUserToken>{

    private WebExtractor<String> tokenStringWebExtractor;
    private final static SNSUserTokenCodec CODEC = new SNSUserTokenCodec();

    public SNSUserTokenWebExtractor() {
        super();
        this.tokenStringWebExtractor = CompositeTokenValueWebExtractor.of(
            new HeaderTokenValueWebExtractor(WebUtils.HEADER_SNS_USER_TOKEN),
            new CookieTokenValueWebExtractor(WebUtils.COOKIE_SNS_USER_TOKEN)
        );
    }

    @Override
    public SNSUserToken extract(MethodParameter parameter, NativeWebRequest webRequest) {
        String tokenString = tokenStringWebExtractor.extract(parameter, webRequest);
        return parseToken(tokenString);
    }

    @Override
    public SNSUserToken extract(HttpServletRequest servletRequest) {
        String tokenString = tokenStringWebExtractor.extract(servletRequest);
        return parseToken(tokenString);
    }
    public SNSUserToken parseToken(String tokenString) {
        if (ObjectUtils.isEmpty(tokenString)) {
            return null;
        }
        SNSUserToken snsUserToken;
        try {
            snsUserToken = CODEC.decode(tokenString);
            snsUserToken.verify();
        } catch (TokenException e) {
            log.warn("SNSUserToken extract exception: {}", tokenString, e);
            return null;
        } catch (Exception e) {
            log.warn("SNSUserToken extract failure: {}", tokenString, e);
            return null;
        }
        return snsUserToken;
    }
}
