package org.beast.security.graphql.resolver;


import graphql.GraphQLContext;
import graphql.schema.DataFetchingEnvironment;
import org.apache.commons.lang3.ArrayUtils;
import org.beast.security.core.*;
import org.beast.security.core.annotation.SNSUserTokenValue;
import org.beast.web.util.WebUtils;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.security.access.AccessDeniedException;

import java.util.Objects;

public class SNSUserTokenArgumentResolver extends AbstractTokenArgumentResolver<SNSUserTokenValue> {


    public SNSUserTokenArgumentResolver() {
        super(SNSUserTokenValue.class);
    }

    @Override
    public boolean supportsParameter(@NonNull MethodParameter parameter) {
        return AnnotatedElementUtils.hasAnnotation(parameter.getParameter(), SNSUserTokenValue.class);
    }

    @Override
    public SNSUserTokenValue resolveAnnotation(MethodParameter parameter) {
        return AnnotatedElementUtils.findMergedAnnotation(parameter.getParameter(), SNSUserTokenValue.class);
    }

    @Override
    public boolean isRequired(SNSUserTokenValue annotation) {
        return annotation.required();
    }


    @Override
    protected void handleMissingValue(MethodParameter parameter) {
        throw new AccessDeniedException("SNSUserToken is required");
    }



    @Nullable
    @Override
    protected Object resolveValue(SNSUserTokenValue annotation, MethodParameter parameter, DataFetchingEnvironment environment) {
        GraphQLContext context = environment.getGraphQlContext();
        SNSUserToken snsUserToken = context.get(WebUtils.HEADER_X_SNS_U_TOKEN);
        if (Objects.isNull(snsUserToken)) {
            return null;
        }
        SNSType[] requiredTypes = annotation.type();
        if (requiredTypes.length > 0) {
            if (!ArrayUtils.contains(requiredTypes, snsUserToken.getType())) {
                return null;
            }
        }
        if (String.class.isAssignableFrom(parameter.getNestedParameterType())) {
            return switch (snsUserToken.getType()) {
                case WECHAT_WEAPP -> ((WechatWeappSNSUserToken)snsUserToken).getOpenid();
                case WECHAT_OFFIACCOUNT -> ((WechatOffiAccountSNSUserToken)snsUserToken).getOpenid();
                case BYTEDANCE_BYTEAPP -> ((BytedanceByteappSNSUserToken)snsUserToken).getOpenid();
                default -> null;
            };
        } else {
            return snsUserToken;
        }

    }

}
