package org.beast.security.graphql.interceptor;

import com.google.common.collect.Maps;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import lombok.extern.slf4j.Slf4j;
import org.beast.security.core.SNSUserToken;
import org.beast.security.core.codec.SNSUserTokenCodec;
import org.beast.security.core.exception.TokenException;
import org.beast.web.util.WebUtils;
import org.springframework.graphql.server.WebGraphQlInterceptor;
import org.springframework.graphql.server.WebGraphQlRequest;
import org.springframework.graphql.server.WebGraphQlResponse;
import org.springframework.http.HttpHeaders;
import org.springframework.util.ObjectUtils;
import reactor.core.publisher.Mono;

import java.util.Map;
import java.util.Objects;

@Slf4j
public class SecurityWebGraphQlInterceptor implements WebGraphQlInterceptor {

    private final static SNSUserTokenCodec CODEC = new SNSUserTokenCodec();
//    private ServerCookieDecoder decoder = ServerCookieDecoder.LAX;
    @Override
    public Mono<WebGraphQlResponse> intercept(WebGraphQlRequest request, Chain chain) {
        request.configureExecutionInput((executionInput, builder) -> {
            final Map<String, Object> context = Maps.newHashMap();
            var headers = request.getHeaders();
            if (headers.containsKey(WebUtils.HEADER_X_SNS_U_TOKEN)) {
                String tokenString = headers.getFirst(WebUtils.HEADER_X_SNS_U_TOKEN);
                SNSUserToken snsUserToken = parseToken(tokenString);
                if (Objects.nonNull(snsUserToken)) {
                    context.put(WebUtils.HEADER_X_SNS_U_TOKEN, snsUserToken);
                }
            }
            return builder.graphQLContext(context).build();
        });
        return chain.next(request);
    }

    public SNSUserToken parseToken(String tokenString) {
        if (ObjectUtils.isEmpty(tokenString)) {
            return null;
        }
        SNSUserToken snsUserToken;
        try {
            snsUserToken = CODEC.decode(tokenString);
            snsUserToken.verify();
        } catch (TokenException e) {
            log.warn("SNSUserToken extract exception: {}", tokenString, e);
            return null;
        } catch (Exception e) {
            log.warn("SNSUserToken extract failure: {}", tokenString, e);
            return null;
        }
        return snsUserToken;
    }
}
