package org.beast.security.core;

import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.beast.security.core.exception.TokenExpiredException;

import java.time.Instant;
import java.util.Objects;

/**
 * UserToken 与  SNSUserToken 是共存的
 *
 * UserToken：用户身份
 * SNSUserToken：社交平台的身份， （在一个存储环境下没特殊会覆盖， 注意避免共存）
 *
 */
@ToString
@JsonTypeInfo(
        use = JsonTypeInfo.Id.NAME,
        include = JsonTypeInfo.As.EXISTING_PROPERTY,
        property = "type",
        visible = true
)
@JsonSubTypes({
    @JsonSubTypes.Type(value = AlipayAliappSNSToken.class, name = "ALIPAY_ALIAPP"),
    @JsonSubTypes.Type(value = WechatWeappSNSUserToken.class, name = "WECHAT_WEAPP"),
    @JsonSubTypes.Type(value = WechatOffiaccountSNSUserToken.class, name = "WECHAT_OFFIACCOUNT"),
    @JsonSubTypes.Type(value = WechatUnionSNSUserToken.class, name = "WECHAT_UNION"),
    @JsonSubTypes.Type(value = BytedanceByteappSNSUserToken.class, name = "BYTEDANCE_BYTEAPP"),
})
@Getter
@Setter
public class SNSUserToken implements TokenNamed {

    public static final String NAME = "SNSUserToken";

    private SNSType type;

    /**
     * SNS应用id
     * - 微信公众号appid
     * - 微信小程序appid
     * - 支付宝应用appid
     */
    private String appid;



    /**
     * 于过期
     */
    private Instant expiresAt;

    /**
     * 于发行
     */
    private Instant issuedAt;

    protected SNSUserToken(SNSType type) {
        this.type = type;
    }

    public String name() {
        return NAME;
    }

    private boolean verifyExp(Instant exp, Instant now) {
        return Objects.nonNull(exp) && exp.isAfter(now);
    }
    public void verify() {
        Instant now = Instant.now();
        if (!this.verifyExp(this.expiresAt, now)) {
            String msg = "token expired at " + this.getExpiresAt() + ". Current time: " + now;
            throw new TokenExpiredException(msg);
        };
    }

}
