package org.beast.security.core.codec;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.SignatureAlgorithm;
import org.beast.security.core.EmailToken;
import org.beast.security.core.util.KeySpecUtils;

import java.security.Key;
import java.time.Instant;
import java.util.Map;

public class EmailTokenCodec implements TokenCodec<EmailToken>{
    private static final String PRI_RESOURCE = "classpath:token/email-token-private.der";
    private static final String PUB_RESOURCE = "classpath:token/email-token-public.der";
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.RS256;

    private static final class CLAIM_NAMES {
        private static final String EMAIL = "email";
        private static final String ISSUED_AT = "iss";
        private static final String EXPIRES_AT = "exp";
    }

    @Override
    public String encode(EmailToken token) {
        return EmailTokenCodec.TokenBuilder.getInstance().compact(token);
    }

    @Override
    public EmailToken decode(String tokenString) {
        return EmailTokenCodec.TokenParser.getInstance().parse(tokenString);
    }

    public static class TokenParser extends AbstractJWTTokenParser<EmailToken> {

        public TokenParser() {
            this(KeySpecUtils.readPublicKey(PUB_RESOURCE));
        }
        public TokenParser(Key key) {
            super(key);
        }

        @Override
        protected EmailToken read(Claims claims) {
            EmailToken token = new EmailToken();
            token.setEmail(claims.get(CLAIM_NAMES.EMAIL, String.class));
            token.setIssuedAt(Instant.ofEpochMilli(claims.get(CLAIM_NAMES.ISSUED_AT, Long.class)));
            token.setExpiresAt(Instant.ofEpochMilli(claims.get(CLAIM_NAMES.EXPIRES_AT, Long.class)));
            token.verify();
            return token;
        }

        private static final EmailTokenCodec.TokenParser INSTANCE = new EmailTokenCodec.TokenParser();

        public static EmailTokenCodec.TokenParser getInstance() {
            return INSTANCE;
        }
    }

    public static class TokenBuilder extends AbstractJWTTokenBuilder<EmailToken> {

        public TokenBuilder() {
            this(KeySpecUtils.readPrivateKey(PRI_RESOURCE),SIGNATURE_ALGORITHM);
        }

        public TokenBuilder(Key key, SignatureAlgorithm algorithm) {
            super(key, algorithm);
        }

        @Override
        protected void write(Map<String, Object> claims, EmailToken token) {
            claims.put(CLAIM_NAMES.EMAIL, token.getEmail());
            claims.put(CLAIM_NAMES.ISSUED_AT, token.getIssuedAt().toEpochMilli());
            claims.put(CLAIM_NAMES.EXPIRES_AT, token.getExpiresAt().toEpochMilli());
        }

        private static final EmailTokenCodec.TokenBuilder INSTANCE = new EmailTokenCodec.TokenBuilder();

        public static EmailTokenCodec.TokenBuilder getInstance() {
            return INSTANCE;
        }
    }
}
