package org.beast.security.core.auth.permission;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.beast.security.core.auth.Permission;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.List;
import java.util.Set;

/**
 *
 * 这边仅代表操作
 *
 * 操作 和 资源限制 应分离
 * 更利于筛选操作后  通过资源限制 来查询且筛选数据
 *
 *
 * 建议表达式:  [资源]:[子资源..]:[操作]
 *
 * ESC:TAG:ADD    ESC:标签:添加
 * ESC:DISK:Attach ESC:磁盘:挂载
 * ESC:Instance:Describe    ESC:实例:Describe  实例描述信息
 * ESC:InstanceStatus:Describe    ESC:实例:Describe  实例状态描述信息
 * ESC:*:Describe ESC只读
 *
 * 资源限制独立 操作验证完成后 进行资源的过滤 更利于实现
 */
public class WildcardPermission implements Permission, Serializable {

    //通配
    protected static final String WILDCARD_TOKEN = "*";
    //分割
    protected static final String PART_DIVIDER_TOKEN = ":";
    protected static final String SUBPART_DIVIDER_TOKEN = ",";


    private List<Set<String>> parts;


    protected WildcardPermission() {

    }

    public WildcardPermission(String wildcardString) {
        setParts(wildcardString);
    }


    protected void setParts(String wildcardString) {
        if (ObjectUtils.isEmpty(wildcardString)) {
            throw new IllegalArgumentException("wildcard string cannot be empty");
        }

        Iterable<String> parts = Splitter.on(PART_DIVIDER_TOKEN).split(wildcardString);
        this.parts = Lists.newArrayList();
        parts.forEach(part -> {
            Set<String> subparts = Sets.newHashSet(part.split(SUBPART_DIVIDER_TOKEN));
            if (subparts.isEmpty()) {
                throw new IllegalArgumentException("Wildcard string cannot contain parts with only dividers");
            }
            this.parts.add(subparts);
        });
        if (this.parts.isEmpty()) {
            throw new IllegalArgumentException("Wildcard string cannot contain parts with only dividers");
        }
    }

    protected List<Set<String>> getParts() {
        return parts;
    }

    protected void setParts(List<Set<String>> parts) {
        this.parts = parts;
    }


    @Override
    public boolean implies(Permission p) {
        if (!(p instanceof WildcardPermission)) {
            return false;
        }
        WildcardPermission wp = (WildcardPermission) p;

        List<Set<String>> parts = this.getParts();
        List<Set<String>> otherParts = wp.getParts();

        int i = 0;
        for (Set<String> otherPart : otherParts) {
            if (parts.size() > i) {
                Set<String> part = parts.get(i);
                if (!part.contains(WILDCARD_TOKEN) && !part.containsAll(otherPart)) {
                     return false;
                }
            } else {
                return true;
            }
            i++;
        }

        while (i < parts.size()) {
            Set<String> part = parts.get(i);
            if (!part.contains(WILDCARD_TOKEN)) {
                return false;
            }
            i++;
        }
        return true;
    }

    @Override
    public String toString() {
        return Joiner.on(PART_DIVIDER_TOKEN).join(parts.stream().map(part -> {
            return Joiner.on(SUBPART_DIVIDER_TOKEN).join(part);
        }).iterator());
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof WildcardPermission) {
            WildcardPermission wp = (WildcardPermission) o;
            return parts.equals(wp.parts);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return parts.hashCode();
    }
}
