package org.beast.security.core.codec;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.SignatureAlgorithm;
import org.beast.security.core.UserToken;
import org.beast.security.core.util.KeySpecUtils;

import java.security.Key;
import java.time.Instant;
import java.util.Map;

public class UserTokenCodec implements TokenCodec<UserToken>{
    public static final String PRI_RESOURCE = "classpath:token/user-token-private.der";
    public static final String PUB_RESOURCE = "classpath:token/user-token-public.der";
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.RS256;

    private static final class CLAIM_NAMES {
        public static final String APP = "app";
        public static final String UID = "uid";
        public static final String ISSUED_AT = "iss";
        public static final String EXPIRES_AT = "exp";
    }

    @Override
    public String encode(UserToken token) {
        return TokenBuilder.getInstance().compact(token);
    }

    @Override
    public UserToken decode(String tokenString) {
        return TokenParser.getInstance().parse(tokenString);
    }

    public static class TokenParser extends AbstractJWTTokenParser<UserToken> {

        public TokenParser() {
            this(KeySpecUtils.readPublicKey(PUB_RESOURCE));
        }
        public TokenParser(Key key) {
            super(key);
        }

        @Override
        protected UserToken read(Claims claims) {
            UserToken token = new UserToken();
            String appId = claims.get(CLAIM_NAMES.APP, String.class);
            Long uid = claims.get(CLAIM_NAMES.UID, Long.class);
            Instant issuedAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.ISSUED_AT, Long.class));
            Instant expiresAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.EXPIRES_AT, Long.class));
            token.setAppId(appId);
            token.setUid(uid);
            token.setIssuedAt(issuedAt);
            token.setExpiresAt(expiresAt);
            return token;
        }

        private static final TokenParser INSTANCE = new TokenParser();

        public static TokenParser getInstance() {
            return INSTANCE;
        }
    }

    public static class TokenBuilder extends AbstractJWTTokenBuilder<UserToken> {

        public TokenBuilder() {
            this(KeySpecUtils.readPrivateKey(PRI_RESOURCE),SIGNATURE_ALGORITHM);
        }

        public TokenBuilder(Key key, SignatureAlgorithm algorithm) {
            super(key, algorithm);
        }

        @Override
        protected void write(Map<String, Object> claims, UserToken token) {
            claims.put(CLAIM_NAMES.APP, token.getAppId());
            claims.put(CLAIM_NAMES.UID, token.getUid());
            claims.put(CLAIM_NAMES.ISSUED_AT, token.getIssuedAt().toEpochMilli());
            claims.put(CLAIM_NAMES.EXPIRES_AT, token.getExpiresAt().toEpochMilli());
        }

        private static final TokenBuilder INSTANCE = new TokenBuilder();

        public static TokenBuilder getInstance() {
            return INSTANCE;
        }
    }
}