package org.beast.security.core.codec;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.SignatureAlgorithm;
import org.beast.security.core.MobileToken;
import org.beast.security.core.UserToken;
import org.beast.security.core.util.KeySpecUtils;

import java.security.Key;
import java.time.Instant;
import java.util.Map;

public class MobileTokenCodec implements TokenCodec<MobileToken>{

    private static final String PRI_RESOURCE = "classpath:token/mobile-token-private.der";
    private static final String PUB_RESOURCE = "classpath:token/mobile-token-public.der";
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.RS256;

    private static final class CLAIM_NAMES {
        private static final String VERIFY_MODE = "verify-mode";
        private static final String MOBILE = "mobile";
        private static final String ISSUED_AT = "iss";
        private static final String EXPIRES_AT = "exp";
    }

    @Override
    public String encode(MobileToken token) {
        return TokenBuilder.getInstance().compact(token);
    }

    @Override
    public MobileToken decode(String tokenString) {
        return TokenParser.getInstance().parse(tokenString);
    }

    public static class TokenParser extends AbstractJWTTokenParser<MobileToken> {

        public TokenParser() {
            this(KeySpecUtils.readPublicKey(PUB_RESOURCE));
        }
        public TokenParser(Key key) {
            super(key);
        }

        @Override
        protected MobileToken read(Claims claims) {
            MobileToken token = new MobileToken();
            String verifyModeString = claims.get(CLAIM_NAMES.VERIFY_MODE, String.class);
            token.setVerifyMode(MobileToken.VerifyMode.valueOf(verifyModeString));
            token.setMobile(claims.get(CLAIM_NAMES.MOBILE, String.class));
            token.setIssuedAt(Instant.ofEpochMilli(claims.get(CLAIM_NAMES.ISSUED_AT, Long.class)));
            token.setExpiresAt(Instant.ofEpochMilli(claims.get(CLAIM_NAMES.EXPIRES_AT, Long.class)));
            token.verify();
            return token;
        }

        private static final MobileTokenCodec.TokenParser INSTANCE = new MobileTokenCodec.TokenParser();

        public static MobileTokenCodec.TokenParser getInstance() {
            return INSTANCE;
        }
    }

    public static class TokenBuilder extends AbstractJWTTokenBuilder<MobileToken> {

        public TokenBuilder() {
            this(KeySpecUtils.readPrivateKey(PRI_RESOURCE),SIGNATURE_ALGORITHM);
        }

        public TokenBuilder(Key key, SignatureAlgorithm algorithm) {
            super(key, algorithm);
        }

        @Override
        protected void write(Map<String, Object> claims, MobileToken token) {
            claims.put(CLAIM_NAMES.VERIFY_MODE, token.getVerifyMode());
            claims.put(CLAIM_NAMES.MOBILE, token.getMobile());
            claims.put(CLAIM_NAMES.ISSUED_AT, token.getIssuedAt().toEpochMilli());
            claims.put(CLAIM_NAMES.EXPIRES_AT, token.getExpiresAt().toEpochMilli());
        }

        private static final MobileTokenCodec.TokenBuilder INSTANCE = new MobileTokenCodec.TokenBuilder();

        public static MobileTokenCodec.TokenBuilder getInstance() {
            return INSTANCE;
        }
    }
}
