package org.beast.security.core;

import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import lombok.Getter;
import lombok.Setter;
import org.beast.security.core.exception.TokenExpiredException;

import java.time.Instant;

/**
 * UserToken 与  SNSUserToken 是共存的
 *
 * UserToken：用户身份
 * SNSUserToken：社交平台的身份， （在一个存储环境下没特殊会覆盖， 注意避免共存）
 *
 */
@JsonTypeInfo(
        use = JsonTypeInfo.Id.NAME,
        include = JsonTypeInfo.As.EXISTING_PROPERTY,
        property = "type",
        visible = true
)
@JsonSubTypes({
    @JsonSubTypes.Type(value = WechatWeappSNSUserToken.class, name = "WECHAT_WEAPP"),
    @JsonSubTypes.Type(value = WechatOffiAccountSNSUserToken.class, name = "WECHAT_OFFIACCOUNT"),
    @JsonSubTypes.Type(value = BytedanceByteappSNSUserToken.class, name = "BYTEDANCE_BYTEAPP"),
})
@Getter
@Setter
public class SNSUserToken implements TokenNamed {

    private static final String NAME = "SNSUserToken";

    private SNSTokenType type;

    /**
     * 于发行
     */
    private Instant issuedAt;


    /**
     * 于过期
     */
    private Instant expiresAt;

    protected SNSUserToken(SNSTokenType type) {
        this.type = type;
    }

    public String name() {
        return NAME;
    }

    private boolean verifyExp(Instant exp, Instant now) {
        return exp.isAfter(now);
    }
    public void verify() {
        Instant now = Instant.now();
        if (!this.verifyExp(this.expiresAt, now)) {
            String msg = "token expired at " + this.getExpiresAt() + ". Current time: " + now;
            throw new TokenExpiredException(msg);
        };
    }
}
