package org.beast.route.predicate;

import com.google.common.collect.Lists;
import io.netty.handler.ipfilter.IpFilterRuleType;
import io.netty.handler.ipfilter.IpSubnetFilterRule;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.server.ServerWebExchange;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;

@Slf4j
public class RemoteAddrPredicateFactory extends AbstractPredicateFactory<RemoteAddrPredicateFactory.Config>{

    @Override
    public Predicate<ServerWebExchange> apply(Config config) {
        List<IpSubnetFilterRule> sources = convert(config.sources);
        return new Predicate<ServerWebExchange>() {
            @Override
            public boolean test(ServerWebExchange exchange) {
                InetSocketAddress remoteAddress = resolve(exchange);
                if (remoteAddress != null) {
                    for (IpSubnetFilterRule source : sources) {
                        if (source.matches(remoteAddress)) {
                            return true;
                        }
                    }
                }
                return false;
            }
        };
    }

    @NotNull
    private List<IpSubnetFilterRule> convert(List<String> values) {
        List<IpSubnetFilterRule> sources = new ArrayList<>();
        for (String arg : values) {
            addSource(sources, arg);
        }
        return sources;
    }

    private void addSource(List<IpSubnetFilterRule> sources, String source) {
        if (!source.contains("/")) { // no netmask, add default
            source = source + "/32";
        }

        String[] ipAddressCidrPrefix = source.split("/",2);
        String ipAddress = ipAddressCidrPrefix[0];
        int cidrPrefix = Integer.parseInt(ipAddressCidrPrefix[1]);

        sources.add(new IpSubnetFilterRule(ipAddress, cidrPrefix, IpFilterRuleType.ACCEPT));
    }

    private final int maxTrustedIndex = Integer.MAX_VALUE;
    public static final String X_FORWARDED_FOR = "X-Forwarded-For";

    public InetSocketAddress resolve(ServerWebExchange exchange) {
        List<String> xForwardedValues = extractXForwardedValues(exchange);
        Collections.reverse(xForwardedValues);
        if (!xForwardedValues.isEmpty()) {
            int index = Math.min(xForwardedValues.size(), maxTrustedIndex) - 1;
            return new InetSocketAddress(xForwardedValues.get(index), 0);
        }
        return resolveRemoteAddress(exchange);
    }

    private InetSocketAddress resolveRemoteAddress(ServerWebExchange exchange) {
        return exchange.getRequest().getRemoteAddress();
    }
    private List<String> extractXForwardedValues(ServerWebExchange exchange) {
        List<String> xForwardedValues = exchange.getRequest().getHeaders()
                .get(X_FORWARDED_FOR);
        if (xForwardedValues == null || xForwardedValues.isEmpty()) {
            return Collections.emptyList();
        }
        if (xForwardedValues.size() > 1) {
            log.warn("Multiple X-Forwarded-For headers found, discarding all");
            return Collections.emptyList();
        }
        List<String> values = Arrays.asList(xForwardedValues.get(0).split(", "));
        if (values.size() == 1 && !StringUtils.hasText(values.get(0))) {
            return Collections.emptyList();
        }
        return values;
    }

    @Getter
    @Setter
    @Validated
    public static class Config {

        @NotEmpty
        private List<String> sources = Lists.newArrayList();
    }
}
