package org.beast.risk.engine.configuration;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.beast.risk.engine.HitAssertion;
import org.beast.risk.engine.RiskRadar;
import org.beast.risk.engine.RiskScanner;
import org.beast.risk.engine.policy.EventHandler;
import org.beast.risk.engine.policy.IntentHandler;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.SimpleTypeConverter;
import org.springframework.beans.TypeConverter;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class RiskConfigService extends ApplicationObjectSupport implements InitializingBean {

    private TypeConverter typeConverter = new SimpleTypeConverter();

    private RiskConfigRunnable runnable = new RiskConfigRunnable();

    private RiskDataSource source;

    private RiskRadar radar;

    public RiskConfigService(RiskDataSource source) {
        this.source = source;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.start();
    }

    @Autowired
    public void setRadar(RiskRadar radar) {
        this.radar = radar;
    }

    public void start() {
        runnable.run();
    }



    public void refresh() {
        //刷新配置
        runnable.run();
    }


    public class RiskConfigRunnable implements Runnable {
        @Override
        public void run() {
            try{
                RiskProperties properties = source.readProperties();
                List<IntentHandler> handlers = Lists.newArrayList();
                List<EventHandler> eventHandlers = Lists.newArrayList();
                for (RiskProperties.Intent intentProperty : properties.getIntent()) {
                    IntentHandler handler = new IntentHandler();
                    var assertionProperties = intentProperty.getAssertion();
                    var name = intentProperty.getName();
                    handler.setName(name);
                    handler.setAssertion(instantiate(assertionProperties.getClazz(), assertionProperties.getInitParams()));
                    handlers.add(handler);
                }

                for (RiskProperties.Event eventProperties : properties.getEvent()) {
                    var scannerProperties = eventProperties.getScanner();
                    var name = eventProperties.getName();
                    EventHandler handler = new EventHandler();
                    handler.setName(name);
                    handler.setScanner(instantiate(scannerProperties.getClazz(), scannerProperties.getInitParams()));
                    eventHandlers.add(handler);
                }
                radar.configuration(handlers, eventHandlers);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        private <T> T instantiate(String className, Map<String, String> initParams) throws Exception {
            Class<T> clazz = (Class<T>) Class.forName(className);
            AutowireCapableBeanFactory beanFactory = getApplicationContext().getAutowireCapableBeanFactory();
            T target = beanFactory.createBean(clazz);
            if (Objects.nonNull(initParams)) {
                for (Field field : clazz.getDeclaredFields()) {
                    String fieldName = field.getName();
                    if (initParams.containsKey(fieldName)) {
                        Class<?> fieldType = field.getType();
                        String valueString = initParams.get(fieldName);
                        Object value = typeConverter.convertIfNecessary(valueString, fieldType);
                        ReflectionUtils.makeAccessible(field);
                        ReflectionUtils.setField(field, target, value);
                    }
                }
            }

            return target;
        }
    }
}
