package org.beast.risk.instrument.meter;

import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.stream.StreamSupport;

public class Tags implements Iterable<Tag> {

    private static final Tags EMPTY = new Tags(new Tag[] {});

    private final Tag[] tags;
    private int last;


    private Tags(Tag[] tags) {
        this.tags = tags;
        Arrays.sort(this.tags);
        dedup();
    }

    private void dedup() {
        int n = tags.length;

        if (n == 0 || n == 1) {
            last = n;
            return;
        }

        // index of next unique element
        int j = 0;

        for (int i = 0; i < n - 1; i++)
            if (!tags[i].getKey().equals(tags[i + 1].getKey()))
                tags[j++] = tags[i];

        tags[j++] = tags[n - 1];
        last = j;
    }

    public Tags and(String key, String value) {
        return and(Tag.of(key, value));
    }

    @Override
    public int hashCode() {
        int result = 1;
        for (int i = 0; i < last; i++) {
            result = 31 * result + tags[i].hashCode();
        }
        return result;
    }


    public Tags and(Tag... tags) {
        if (tags == null || tags.length == 0) {
            return this;
        }
        Tag[] newTags = new Tag[last + tags.length];
        System.arraycopy(this.tags, 0, newTags, 0, last);
        System.arraycopy(tags, 0, newTags, last, tags.length);
        return new Tags(newTags);
    }

    public Tags and(Iterable<Tag> tags) {
        if (tags == null || !tags.iterator().hasNext()) {
            return this;
        }

        if (this.tags.length == 0) {
            return Tags.of(tags);
        }

        return and(Tags.of(tags).tags);
    }

    @Override
    public Iterator<Tag> iterator() {
        return new ArrayIterator();
    }

    private class ArrayIterator implements Iterator<Tag> {

        private int currentIndex = 0;

        @Override
        public boolean hasNext() {
            return currentIndex < last;
        }

        @Override
        public Tag next() {
            return tags[currentIndex++];
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("cannot remove items from tags");
        }

    }

    public static Tags of(Iterable<? extends Tag> tags) {
        if (tags == null || !tags.iterator().hasNext()) {
            return Tags.empty();
        }
        else if (tags instanceof Tags) {
            return (Tags) tags;
        }
        else if (tags instanceof Collection) {
            Collection<? extends Tag> tagsCollection = (Collection<? extends Tag>) tags;
            return new Tags(tagsCollection.toArray(new Tag[0]));
        }
        else {
            return new Tags(StreamSupport.stream(tags.spliterator(), false).toArray(Tag[]::new));
        }
    }

    public static Tags of(Tag ...tags) {
        return empty().and(tags);
    }

    public static Tags empty() {
        return EMPTY;
    }
}
