package org.beast.risk.instrument.meter;

import org.beast.risk.instrument.meter.noop.NoopRecentlyCounter;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;

import static java.lang.String.format;

public class MeterRegistry {

    private final Object meterMapLock = new Object();
    private final Map<Meter.Id, Meter> meterMap = new ConcurrentHashMap<>();
    private final AtomicBoolean closed = new AtomicBoolean();
//    public abstract RecentlyCounter newRecentlyCounter(Meter.Id id);

//    public RecentlyCounter recentlyCounter(String name) {
//        return this.recentlyCounter(name, null);
//    }
//    public RecentlyCounter recentlyCounter(String name, String tag) {
//        return RecentlyCounter.builder(name).tag(tag).register(this);
//    }
//
//    public RecentlyCounter recentlyCounter(Meter.Id id) {
//        return registerMeterIfNecessary(RecentlyCounter.class, id, this::newRecentlyCounter, NoopRecentlyCounter::new);
//    }
//
//
//    protected abstract RecentlyCounter newRecentlyCounter(Meter.Id id);

    private Meter getOrCreateMeter(Function<Meter.Id, ? extends Meter> builder, Meter.Id originalId,
                                   Meter.Id mappedId, Function<Meter.Id, ? extends Meter> noopBuilder) {
        var m = meterMap.get(mappedId);
        if (Objects.isNull(m)) {
            synchronized (meterMapLock) {
                m = meterMap.get(mappedId);
                if (Objects.isNull(m)) {
                    m = builder.apply(mappedId);
                    meterMap.put(mappedId, m);
                }
            }
        }
        return m;
    }

    public <M extends Meter> M registerMeterIfNecessary(Class<M> meterClass, Meter.Id id, Function<Meter.Id, M> builder) {
        return registerMeterIfNecessary(meterClass, id, builder, null);
    }
    public <M extends Meter> M registerMeterIfNecessary(Class<M> meterClass, Meter.Id id, Function<Meter.Id, M> builder, Function<Meter.Id, M> noopBuilder) {
        var mappedId = id;

        var m = getOrCreateMeter(builder, id, mappedId, noopBuilder);
        if (!meterClass.isInstance(m)) {
            throw new IllegalArgumentException(
                    format("There is already a registered meter of a different type (%s vs. %s) with the same name: %s",
                            m.getClass().getSimpleName(), meterClass.getSimpleName(), id.getName()));
        }
        return meterClass.cast(m);
    }

    public boolean isClosed() {
        return closed.get();
    }

    public void close() {
        if (closed.compareAndSet(false, true)) {
            synchronized (meterMapLock) {
                for (Meter meter : meterMap.values()) {
                    meter.close();
                }
            }
        }
    }
}
