package org.beast.propagation.instrument.web;

import org.beast.propagation.Propagation;
import org.beast.propagation.context.Context.Extractor;
import org.beast.propagation.context.CurrentContext;
import lombok.extern.slf4j.Slf4j;
import org.reactivestreams.Subscription;
import org.springframework.core.Ordered;
import org.springframework.http.HttpHeaders;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.lang.NonNull;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.CoreSubscriber;
import reactor.core.publisher.Mono;
import reactor.core.publisher.MonoOperator;
import reactor.util.context.Context;

;


@Slf4j
public class PropagationWebFilter implements WebFilter, Ordered {

    private Extractor<HttpHeaders> extractor;
    private CurrentContext currentContext;

    public PropagationWebFilter(Propagation propagation, CurrentContext currentContext) {
        this.extractor = propagation.extractor(new Propagation.Getter<HttpHeaders>() {
            @Override
            public String get(HttpHeaders carrier, String key) {
                return carrier.getFirst(key);
            }

        });
        this.currentContext = currentContext;
    }


    @Override
    public @NonNull Mono<Void> filter(@NonNull ServerWebExchange exchange,@NonNull WebFilterChain chain) {
        return new WebFilterMonoOperator(chain.filter(exchange), exchange);
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE + 4;
    }

    private class WebFilterMonoOperator extends MonoOperator<Void, Void> {

        final ServerWebExchange exchange;

        protected WebFilterMonoOperator(Mono<? extends Void> source, ServerWebExchange exchange) {
            super(source);
            this.exchange = exchange;
        }

        @Override
        public void subscribe(@NonNull CoreSubscriber<? super Void> subscriber) {
            Context context = subscriber.currentContext();
            this.source.subscribe(new WebFilterSubscriber(subscriber, context, findOrCreateScope(context)));
        }

        private CurrentContext.Scope findOrCreateScope(Context c) {
            CurrentContext.Scope scope;
            Object key = CurrentContext.Scope.class;
            if (c.hasKey(key)) {
                scope = c.get(key);
            } else {
                ServerHttpRequest request = exchange.getRequest();
                org.beast.propagation.context.Context context = PropagationWebFilter.this.extractor.extract(request.getHeaders());
                scope = PropagationWebFilter.this.currentContext.newScope(context);
            }
            return scope;
        }

        final class WebFilterSubscriber implements CoreSubscriber<Void> {


            final CoreSubscriber<? super Void> actual;
            final Context context;
            final CurrentContext.Scope scope;


            public WebFilterSubscriber(CoreSubscriber<? super Void> actual, Context context, CurrentContext.Scope scope) {
                this.actual = actual;
                this.context = context.put(CurrentContext.Scope.class, scope);
                this.scope = scope;
            }

            @Override
            public void onSubscribe(Subscription subscription) {
                this.actual.onSubscribe(subscription);
            }

            @Override
            public void onNext(Void aVoid) {

            }

            @Override
            public void onError(Throwable t) {
                this.terminate();
                this.actual.onError(t);
            }

            @Override
            public void onComplete() {
                this.terminate();
                this.actual.onComplete();
            }

            @Override
            public Context currentContext() {
                return this.context;
            }

            private void terminate() {
                this.scope.close();
            }
        }
    }


}
