package org.beast.propagation.shunt.loadbalancer;

import org.beast.propagation.shunt.Shunt;
import org.beast.propagation.shunt.propagation.ShuntContext;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.loadbalancer.Request;
import org.springframework.cloud.loadbalancer.core.DelegatingServiceInstanceListSupplier;
import org.springframework.cloud.loadbalancer.core.ServiceInstanceListSupplier;
import org.springframework.util.CollectionUtils;
import reactor.core.publisher.Flux;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 分流偏好服务实例提供
 */
public class ShuntPreferenceServiceInstanceListSupplier extends DelegatingServiceInstanceListSupplier {

    private final static String GROUP = "group";

    private Shunt shunt;

    public ShuntPreferenceServiceInstanceListSupplier(ServiceInstanceListSupplier delegate, Shunt shunt) {
        super(delegate);
        this.shunt = shunt;
    }

    @Override
    public Flux<List<ServiceInstance>> get() {
        return getDelegate().get().map(this::filteredByShunt);
    }

    @Override
    public Flux<List<ServiceInstance>> get(Request request) {
        return this.get();
    }

    private List<ServiceInstance> filteredByShunt(List<ServiceInstance> serviceInstances) {
        ShuntContext context = shunt.currentShuntContext().get();
        List<String> sequences = Collections.emptyList();
        if (context != null) {
            sequences = context.sequences();
        }
        List<ServiceInstance> filteredInstances = new ArrayList<>();
        if (!CollectionUtils.isEmpty(sequences)) {
            // if the sequence is set
            for (String sequence : sequences) {
                for (ServiceInstance serviceInstance : serviceInstances) {
                    String serverGroup = getGroup(serviceInstance);
                    if (sequence.equals(serverGroup)) {
                        filteredInstances.add(serviceInstance);
                    }
                }
                if (filteredInstances.size() > 0) {
                    return filteredInstances;
                }
            }

        }

        for (ServiceInstance serviceInstance : serviceInstances) {
            String serverGroup = getGroup(serviceInstance);
            if ("standard".equals(serverGroup) || serverGroup == null) {
                filteredInstances.add(serviceInstance);
            }
        }
        return filteredInstances;
    }

    private String getGroup(ServiceInstance serviceInstance) {
        Map<String, String> metadata = serviceInstance.getMetadata();
        if (metadata != null) {
            return metadata.get(GROUP);
        }
        return null;
    }


}
