package org.beast.payment.core;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Sets;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Set;

@Getter @Setter
public class PaymentSymbol implements Serializable {

    protected static final String PART_DIVIDER_TOKEN = PaymentType.PART_DIVIDER_TOKEN;
    protected static final String WILDCARD_TOKEN = "*";
    protected static final String SUBPART_DIVIDER_TOKEN = ",";

    private PayMethod method;

    private Set<String> modes;

    public PaymentSymbol(String symbol) {
        parseSymbol(symbol);
    }

    public PaymentSymbol(PayMethod method, String mode) {
        this.method = method;
        this.modes = Sets.newHashSet(mode);
    }


    public static PaymentSymbol valueOf(PayMethod method, String mode) {
        return new PaymentSymbol(method, mode);
    }

    public static PaymentSymbol valueOf(String symbol) {
        return new PaymentSymbol(symbol);
    }

    private void parseSymbol(String symbolString) {
        if (StringUtils.isEmpty(symbolString)) {
            throw new IllegalArgumentException("symbol string cannot be empty");
        }
        Iterable<String> parts = Splitter.on(PART_DIVIDER_TOKEN).split(symbolString);
        Iterator<String> partsIter = parts.iterator();
        this.method = PayMethod.valueOf(partsIter.next());
        this.modes = Sets.newHashSet(partsIter.next().split(SUBPART_DIVIDER_TOKEN));
    }

    public boolean implies(String symbol) {
        return this.implies(new PaymentSymbol(symbol));
    }
    public boolean implies(PaymentSymbol symbol) {
        if (method != symbol.getMethod()) {
            return false;
        }
        return this.modes.contains(WILDCARD_TOKEN) || this.modes.containsAll(symbol.getModes());
    }
    public boolean implies(PayMethod payMethod, String payMode) {
        return this.implies(new PaymentSymbol(payMethod, payMode));
    }

    @Override
    public String toString() {
        return method + PART_DIVIDER_TOKEN + Joiner.on(SUBPART_DIVIDER_TOKEN).join(this.modes);
    }
}
