package org.beast.payment.channel.wechatv3;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.wechat.pay.contrib.apache.httpclient.auth.Verifier;
import com.wechat.pay.contrib.apache.httpclient.exception.ParseException;
import com.wechat.pay.contrib.apache.httpclient.exception.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.beast.payment.channel.wechatv3.model.WechatNotifyRequest;
import org.beast.payment.channel.wechatv3.model.WechatPayNotify;


import java.io.IOException;
import java.nio.charset.StandardCharsets;

@Slf4j
public class WechatPayNotifyHandler {

    private ObjectMapper mapper;

    private Verifier verifier;

    public WechatPayNotifyHandler(Verifier verifier) {
        this.verifier = verifier;
        this.mapper = new ObjectMapper().setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .setPropertyNamingStrategy(PropertyNamingStrategies.SNAKE_CASE)
                .registerModule(new JavaTimeModule());
    }

    public WechatPayNotify parse(WechatNotifyRequest request) throws IOException, ParseException, ValidationException {

        String body = request.getBody();

        String message = buildMessage(request);
        String serial = request.getSerialNumber();
        String signature = request.getSignature();
        if (!validate(message, serial, signature)) {
            String errorMessage = String
                    .format("解析验签失败：serial=[%s] message=[%s] sign=[%s]", serial, message, signature);
            throw new ValidationException(errorMessage);
        }

        return parseBody(body);
    }

    private boolean validate(String message, String serial, String signature) throws IOException {
        try {
            if (!verifier.verify(serial, message.getBytes(StandardCharsets.UTF_8), signature)) {
                throw verifyFail("serial=[%s] message=[%s] sign=[%s]",
                        serial, message, signature);
            }
        } catch (IllegalArgumentException e) {
            log.warn(e.getMessage());
            return false;
        }
        return true;
    }

    public <T> T decode(String body, Class<T> clazz) throws JsonProcessingException {
        return this.mapper.readValue(body, clazz);
    }
    private WechatPayNotify parseBody(String body) throws ParseException {
        try {
            return this.mapper.readValue(body, WechatPayNotify.class);
        } catch (IOException exception) {
            throw new ParseException("解析body失败, body:" + body, exception);
        }
    }
    protected static IllegalArgumentException verifyFail(String message, Object... args) {
        message = String.format(message, args);
        return new IllegalArgumentException("signature verify fail: " + message);
    }
    protected String buildMessage(WechatNotifyRequest request) throws IOException {
        String timestamp = request.getTimestamp();
        String nonce = request.getNonce();
        String body = request.getBody();

        return timestamp + "\n"
                + nonce + "\n"
                + body + "\n";
    }
}
