package org.beast.payment.channel.wechatv3;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.wechat.pay.contrib.apache.httpclient.Validator;
import com.wechat.pay.contrib.apache.httpclient.WechatPayHttpClientBuilder;
import com.wechat.pay.contrib.apache.httpclient.auth.*;
import com.wechat.pay.contrib.apache.httpclient.cert.CertificatesManager;
import com.wechat.pay.contrib.apache.httpclient.util.AesUtil;
import com.wechat.pay.contrib.apache.httpclient.util.PemUtil;
import lombok.Data;
import lombok.SneakyThrows;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.util.EntityUtils;
import org.beast.payment.channel.AbstractClientFactory;
import org.beast.payment.channel.wechatv3.api.Certificates;
import org.beast.payment.channel.wechatv3.model.PlainCertificateItem;
import org.springframework.util.ObjectUtils;

import javax.annotation.Nullable;
import javax.validation.constraints.NotBlank;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;


public class WechatPayClientFactory extends AbstractClientFactory<WechatPayClient, WechatPayClientFactory.Config> {

    //V3不支持沙箱
    public static final String TARGET = "https://api.mch.weixin.qq.com";

    public WechatPayClientFactory() {

    }

//    public static PrivateKey readPrivateKey(String content) {
//        try {
//            byte[] keyBytes = content.getBytes(StandardCharsets.UTF_8);
//            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
//            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
//            return keyFactory.generatePrivate(keySpec);
//        } catch (Exception e) {
//            throw new UnsupportedOperationException(e);
//        }
//    }

    @SneakyThrows
    @Override
    public WechatPayClient newClient(Config config) {
        String endpoint = config.endpoint;
        if (ObjectUtils.isEmpty(endpoint)) {
            endpoint = TARGET;
        }
        WechatPayHttpClientBuilder builder = WechatPayHttpClientBuilder.create();

        String merchantId = config.merchantId;
        String merchantSerialNumber = config.merchantSerialNumber;
        String apiV3Key = config.apiV3Key;
        PrivateKey merchantPrivateKey = PemUtil.loadPrivateKey(config.merchantPrivateKey);
//        WechatPayHttpClientBuilder certificatesClientBuilder = WechatPayHttpClientBuilder.create()
//                .withMerchant(
//                        merchantId,
//                        merchantSerialNumber,
//                        merchantPrivateKey
//                )
//                .withValidator(response -> true);
//        HttpClient httpClient = certificatesClientBuilder.build();
//        URIBuilder uriBuilder = new URIBuilder(endpoint + "/v3/certificates");
//        HttpGet httpGet = new HttpGet(uriBuilder.build());
//        httpGet.addHeader("Accept", "application/json");
//        HttpResponse response = httpClient.execute(httpGet);
//        String bodyAsString = EntityUtils.toString(response.getEntity());
//        ObjectMapper objectMapper = new ObjectMapper()
//                .setPropertyNamingStrategy(PropertyNamingStrategies.SNAKE_CASE)
//                .registerModule(new JavaTimeModule());
//        Certificates certificates = objectMapper.readValue(bodyAsString, Certificates.class);
//        List<PlainCertificateItem> certificateItems = decryptAndValidate(apiV3Key, certificates, (CloseableHttpResponse)response);
//        if (certificateItems != null) {
//            Optional<PlainCertificateItem> plainCertificateItemOptional = certificateItems.stream().findFirst();
//            if (plainCertificateItemOptional.isPresent()) {
//                String plainCertificate = plainCertificateItemOptional.get().getPlainCertificate();
//                ByteArrayInputStream inputStream = new ByteArrayInputStream(plainCertificate.getBytes(StandardCharsets.UTF_8));
//                builder = builder.withWechatPay(
//                        List.of(PemUtil.loadCertificate(inputStream))
//                );
//            }
//        }

        builder = builder.withMerchant(
                merchantId,
                merchantSerialNumber,
                merchantPrivateKey
        );
        CertificatesManager certificatesManager = CertificatesManager.getInstance();

        certificatesManager.putMerchant(
                merchantId,
                new WechatPay2Credentials(
                        merchantId,
                        new PrivateKeySigner(merchantSerialNumber, merchantPrivateKey)
                ),
                apiV3Key.getBytes(StandardCharsets.UTF_8)
        );

        Verifier verifier = certificatesManager.getVerifier(merchantId);
        builder = builder.withValidator(new WechatPay2Validator(verifier));

        AesUtil aesUtil = new AesUtil(apiV3Key.getBytes(StandardCharsets.UTF_8));

        WechatPrivateKeySigner signer = new WechatPrivateKeySigner(merchantPrivateKey);
        return new WechatPayClient(merchantId, endpoint, signer, verifier, aesUtil, builder);
    }

//    /**
//     * 仅在PayClient加载后才生效
//     */
//    public Verifier getVerifier(String merchantId) throws NotFoundException {
//        return CertificatesManager.getInstance().getVerifier(merchantId);
//    }

    private List<PlainCertificateItem> decryptAndValidate(String apiV3key, Certificates certList, CloseableHttpResponse response) throws GeneralSecurityException, IOException, IOException {
        List<PlainCertificateItem> plainCerts = new ArrayList<>();
        List<X509Certificate> x509Certs = new ArrayList<>();
        AesUtil decryptor = new AesUtil(apiV3key.getBytes(StandardCharsets.UTF_8));
        for (Certificates.CertificateItem item : certList.getData()) {
            PlainCertificateItem plainCert = new PlainCertificateItem(
                    item.getSerialNo(), item.getEffectiveTime(), item.getExpireTime(),
                    decryptor.decryptToString(
                            item.getEncryptCertificate().getAssociatedData().getBytes(StandardCharsets.UTF_8),
                            item.getEncryptCertificate().getNonce().getBytes(StandardCharsets.UTF_8),
                            item.getEncryptCertificate().getCiphertext()));
            System.out.println(plainCert.getPlainCertificate());

            ByteArrayInputStream inputStream = new ByteArrayInputStream(plainCert.getPlainCertificate().getBytes(StandardCharsets.UTF_8));
            X509Certificate x509Cert = PemUtil.loadCertificate(inputStream);
            plainCerts.add(plainCert);
            x509Certs.add(x509Cert);
        }

        //从下载的证书中，获取对报文签名的私钥所对应的证书，并进行验签来验证证书准确
        Verifier verifier = new CertificatesVerifier(x509Certs);
        Validator validator = new WechatPay2Validator(verifier);
        boolean isCorrectCert = validator.validate(response);
        System.out.println(isCorrectCert ? "=== validate success ===" : "=== validate failed ===");
        return isCorrectCert ? plainCerts : null;
    }

    @Data
    public static final class Config {
        @Nullable
        private String endpoint;
        //商户号
        @NotBlank
        private String merchantId;
        //商户API证书的证书序列号。
        @NotBlank
        private String merchantSerialNumber;
        //商户API私钥
        @NotBlank
        private String merchantPrivateKey;
        // API V3密钥
        @NotBlank
        private String apiV3Key;

    }
}
