package org.beast.payment.channel.wechatv3;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.wechat.pay.contrib.apache.httpclient.WechatPayHttpClientBuilder;
import com.wechat.pay.contrib.apache.httpclient.auth.Verifier;
import com.wechat.pay.contrib.apache.httpclient.exception.ParseException;
import com.wechat.pay.contrib.apache.httpclient.exception.ValidationException;
import com.wechat.pay.contrib.apache.httpclient.util.AesUtil;
import feign.Feign;
import feign.httpclient.ApacheHttpClient;
import feign.jackson.JacksonDecoder;
import feign.jackson.JacksonEncoder;
import lombok.experimental.Delegate;
import lombok.extern.slf4j.Slf4j;
import org.beast.payment.channel.wechatv3.api.WechatV3PayHttpClient;
import org.beast.payment.channel.wechatv3.model.WechatNotifyRequest;
import org.beast.payment.channel.wechatv3.model.WechatPayNotify;
import org.springframework.cloud.openfeign.support.SpringMvcContract;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;


@Slf4j
public class WechatPayClient {

    private String merchantId;

    private WechatPrivateKeySigner signer;

    //前面验证
    private Verifier verifier;
//
    private AesUtil aesUtil;
    @Delegate
    private WechatV3PayHttpClient httpClient;

    private WechatPayNotifyHandler notifyHandler;



    public WechatPayClient(
            String merchantId,
            String endpoint,
            WechatPrivateKeySigner signer,
            Verifier verifier,
            AesUtil aesUtil,
            WechatPayHttpClientBuilder builder
    ) {
        this.merchantId = merchantId;
        this.signer = signer;
        this.verifier = verifier;
        this.notifyHandler = new WechatPayNotifyHandler(verifier);
        this.aesUtil = aesUtil;
        ApacheHttpClient httpClient = new ApacheHttpClient(builder.build());
        ObjectMapper mapper = new ObjectMapper().setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .setPropertyNamingStrategy(PropertyNamingStrategies.SNAKE_CASE)
                .registerModule(new JavaTimeModule());
        this.httpClient = Feign.builder()
                .client(httpClient)
                .contract(new SpringMvcContract())
                .encoder(new JacksonEncoder(mapper))
                .decoder(new JacksonDecoder(mapper))
                .target(WechatV3PayHttpClient.class, endpoint);
    }

    public String getMerchantId() {
        return merchantId;
    }

    public String sign(String message) {
        return signer.sign(message);
    }

    public final boolean validate(
            String serial,
            String message,
            String signature
    ) {
        try {
            if (!verifier.verify(serial, message.getBytes(StandardCharsets.UTF_8), signature)) {
                throw verifyFail("serial=[%s] message=[%s] sign=[%s]",
                        serial, message, signature);
            }
        } catch (IllegalArgumentException e) {
            log.warn(e.getMessage());
            return false;
        }
        return true;
    }

    public WechatPayNotify parsePayNotify(WechatNotifyRequest request) throws ValidationException, IOException, ParseException {
        return notifyHandler.parse(request);
    }

    public <T> T decryptData(WechatPayNotify notify, Class<T> clazz) throws ParseException, JsonProcessingException {
        WechatPayNotify.Resource resource = notify.getResource();
        String getAssociateddData = "";
        if (resource.getAssociatedData() != null) {
            getAssociateddData = resource.getAssociatedData();
        }
        byte[] associatedData = getAssociateddData.getBytes(StandardCharsets.UTF_8);
        byte[] nonce = resource.getNonce().getBytes(StandardCharsets.UTF_8);
        String ciphertext = resource.getCiphertext();
        String decryptData;
        try {
            decryptData = aesUtil.decryptToString(associatedData, nonce, ciphertext);
        } catch (GeneralSecurityException e) {
            throw new ParseException("AES解密失败，resource：" + resource, e);
        }
        return notifyHandler.decode(decryptData, clazz);
    }

    protected static IllegalArgumentException verifyFail(String message, Object... args) {
        message = String.format(message, args);
        return new IllegalArgumentException("signature verify fail: " + message);
    }
}
