package org.beast.payment.channel.byteapp;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import feign.Feign;
import feign.jackson.JacksonDecoder;
import feign.jackson.JacksonEncoder;
import lombok.SneakyThrows;
import org.beast.payment.channel.byteapp.api.ByteAppPayHttpClient;
import org.beast.payment.channel.byteapp.api.CreateOrderInput;
import org.beast.payment.channel.byteapp.api.CreateOrderOutput;
import org.beast.payment.channel.byteapp.exception.ByteAppPayException;
import org.beast.payment.channel.byteapp.model.ByteAppNotifyRequest;
import org.beast.payment.channel.byteapp.model.ByteAppPayNotify;
import org.springframework.cloud.openfeign.support.SpringMvcContract;

import java.util.Map;

public class ByteAppPayClient {

    private String appId;

    private ByteAppKeySigner signer;

    private ByteAppNotifyVerifier notifyVerifier;

    private ByteAppPayNotifyHandler notifyHandler;

    private ByteAppPayHttpClient httpClient;

    private ObjectMapper mapper;



    private static final String ENDPOINT = "https://developer.toutiao.com";


    public ByteAppPayClient(String appId, String key, String token) {
        this.appId = appId;
        this.signer = new ByteAppKeySigner(key);
        this.notifyVerifier = new ByteAppNotifyVerifier(token);
        this.notifyHandler = new ByteAppPayNotifyHandler(notifyVerifier);
        this.mapper = new ObjectMapper()
                .setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .setPropertyNamingStrategy(PropertyNamingStrategies.SNAKE_CASE)
                .registerModule(new JavaTimeModule());
        this.httpClient = Feign.builder()
                .contract(new SpringMvcContract())
                .encoder(new JacksonEncoder(mapper))
                .decoder(new JacksonDecoder(mapper))
                .target(ByteAppPayHttpClient.class, ENDPOINT);
    }


    public CreateOrderOutput createOrder(
            CreateOrderInput input
    ) {
        Map<String, Object> params = buildRequestParams(input);
        return httpClient.createOrder(params);
    }

    public ByteAppPayNotify parsePayNotify(ByteAppNotifyRequest request) throws ByteAppPayException, JsonProcessingException {
        return notifyHandler.parse(request);
    }

    @SneakyThrows
    private Map<String, Object> buildRequestParams(Object input) {
        TreeNode node = mapper.valueToTree(input);
        Map<String, Object> params = mapper.treeToValue(node, Map.class);
        String sign = signer.sign(params);
        params.put("app_id", appId);
        params.put("sign", sign);
        return params;
    }

}
