package org.beast.pay.data;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Objects;
import java.util.Set;


public class PaymentSymbol implements Serializable {

    protected static final String PART_DIVIDER_TOKEN = PaymentMethod.PART_DIVIDER_TOKEN;
    protected static final String WILDCARD_TOKEN = "*";
    protected static final String SUBPART_DIVIDER_TOKEN = ",";

    private String type;

    private Set<String> modes;

    public PaymentSymbol(String symbol) {
        parseSymbol(symbol);
    }

    public PaymentSymbol(String type, String mode) {
        this.type = type;
        this.modes = Sets.newHashSet(mode);
    }


    public static PaymentSymbol valueOf(String type, String mode) {
        return new PaymentSymbol(type, mode);
    }

//    @JsonCreator
    public static PaymentSymbol valueOf(String method) {
        return new PaymentSymbol(method);
    }

    private void parseSymbol(String symbolString) {
        if (StringUtils.isEmpty(symbolString)) {
            throw new IllegalArgumentException("symbol string cannot be empty");
        }
        Iterable<String> parts = Splitter.on(PART_DIVIDER_TOKEN).split(symbolString);
        Iterator<String> partsIter = parts.iterator();
        this.type = partsIter.next();
        this.modes = Sets.newHashSet(partsIter.next().split(SUBPART_DIVIDER_TOKEN));
    }

    public boolean implies(String symbol) {
        return this.implies(new PaymentSymbol(symbol));
    }
    public boolean implies(PaymentSymbol symbol) {
        if (!Objects.equals(type, symbol.type)) {
            return false;
        }
        return this.modes.contains(WILDCARD_TOKEN) || this.modes.containsAll(symbol.modes);
    }
    public boolean implies(String type, String mode) {
        return this.implies(new PaymentSymbol(type, mode));
    }

//    @JsonValue
    @Override
    public String toString() {
        return type + PART_DIVIDER_TOKEN + Joiner.on(SUBPART_DIVIDER_TOKEN).join(this.modes);
    }
}
