package org.beast.graphql.configuration;

import org.beast.graphql.resolver.PageableMethodArgumentResolver;
import org.beast.graphql.resolver.SortMethodArgumentResolver;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcAutoConfiguration;
import org.springframework.boot.convert.ApplicationConversionService;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.expression.BeanFactoryResolver;
import org.springframework.core.convert.ConversionService;
import org.springframework.expression.BeanResolver;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.graphql.data.GraphQlArgumentBinder;
import org.springframework.graphql.data.method.HandlerMethodArgumentResolver;
import org.springframework.graphql.data.method.HandlerMethodArgumentResolverComposite;
import org.springframework.graphql.data.method.annotation.support.*;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.concurrent.*;

@AutoConfigureAfter({ WebMvcAutoConfiguration.class })
@Configuration(proxyBeanMethods = false)
public class BeastGraphqlAutoConfiguration {

    private final ListableBeanFactory beanFactory;

    public BeastGraphqlAutoConfiguration(ListableBeanFactory beanFactory) {
        this.beanFactory = beanFactory;
    }

    @Bean
    public PageableMethodArgumentResolver pageableMethodArgumentResolver() {
        return new PageableMethodArgumentResolver();
    }

    @Bean
    public SortMethodArgumentResolver sortMethodArgumentResolver() {
        return new SortMethodArgumentResolver();
    }

    @ConditionalOnMissingClass
    @Bean
    public Executor graphqlExecutor() {
        int nThreads = 5;
        var threadFactory = new CustomizableThreadFactory("pool-graphql-async-threads");
        return new ThreadPoolExecutor(nThreads, nThreads,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>(), threadFactory);
    }
    //TODO 这边忘了修改了什么 不知道会不会有问题，可能是自定义HandlerMethodArgumentResolver的支持,新版可能支持了
    @Bean
    public AnnotatedControllerConfigurer annotatedControllerConfigurer(
            ObjectProvider<ConversionService> conversionService,
            ObjectProvider<HandlerMethodArgumentResolver> provider,
            @Qualifier("graphqlExecutor") Executor executor
    ) {
//        GenericHttpMessageConverter<Object> jsonMessageConverter = getJsonConverter(converters);
        AnnotatedControllerConfigurer registrar = new AnnotatedControllerConfigurer() {
            private final boolean springDataPresent = ClassUtils.isPresent(
                    "org.springframework.data.projection.SpelAwareProxyProjectionFactory",
                    AnnotatedControllerConfigurer.class.getClassLoader());
            private final boolean springSecurityPresent = ClassUtils.isPresent(
                    "org.springframework.security.core.context.SecurityContext",
                    AnnotatedControllerConfigurer.class.getClassLoader());


            @Override
            public void afterPropertiesSet() {
                //TODO 重设了afterPropertiesSet
                super.afterPropertiesSet();
                Field field = ReflectionUtils.findField(AnnotatedControllerConfigurer.class, "argumentResolvers");
                Assert.notNull(field, "Reflection argumentResolvers field is not exists ");
                ReflectionUtils.makeAccessible(field);

                HandlerMethodArgumentResolverComposite resolvers = new HandlerMethodArgumentResolverComposite();

                // Annotation based
                if (springDataPresent) {
                    // Must be ahead of ArgumentMethodArgumentResolver
                    resolvers.addResolver(new ProjectedPayloadMethodArgumentResolver(obtainApplicationContext()));
                }
                //TODO 新版没这个要查一下
//                resolvers.addResolver(new ArgumentMapMethodArgumentResolver());
                GraphQlArgumentBinder argumentBinder = new GraphQlArgumentBinder(conversionService.getIfAvailable(DefaultFormattingConversionService::new));
                resolvers.addResolver(new ArgumentMethodArgumentResolver(argumentBinder));
                resolvers.addResolver(new ArgumentsMethodArgumentResolver(argumentBinder));
                resolvers.addResolver(new ContextValueMethodArgumentResolver());
                resolvers.addResolver(new LocalContextValueMethodArgumentResolver());

                //TODO 可能实现了这个
                provider.forEach(resolvers::addResolver);

                // Type based
                resolvers.addResolver(new DataFetchingEnvironmentMethodArgumentResolver());
                resolvers.addResolver(new DataLoaderMethodArgumentResolver());
                if (springSecurityPresent) {
                    resolvers.addResolver(new PrincipalMethodArgumentResolver());
                    BeanResolver beanResolver = new BeanFactoryResolver(obtainApplicationContext());
                    resolvers.addResolver(new AuthenticationPrincipalArgumentResolver(beanResolver));
                }
//                if (KotlinDetector.isKotlinPresent()) {
//                    argumentResolvers.addResolver(new ContinuationHandlerMethodArgumentResolver());
//                }

                // This works as a fallback, after other resolvers
                resolvers.addResolver(new SourceMethodArgumentResolver());
                ReflectionUtils.setField(field, this, resolvers);
            }
        };
        //TODO 扩展用于异步查询
        registrar.setExecutor(executor);
        registrar.addFormatterRegistrar((registry) -> ApplicationConversionService.addBeans(registry, this.beanFactory));
        return registrar;
    }

}
