package org.beast.graphql.resolver;

import graphql.schema.DataFetchingEnvironment;
import org.springframework.core.MethodParameter;
import org.springframework.graphql.data.method.HandlerMethodArgumentResolver;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;

import java.lang.annotation.Annotation;

public abstract class AbstractTokenArgumentResolver<A extends Annotation> implements HandlerMethodArgumentResolver {

    private Class<A> annotationClass;

    public AbstractTokenArgumentResolver(Class<A> annotationClass) {
        this.annotationClass = annotationClass;
    }

    @Override
    public boolean supportsParameter(@NonNull MethodParameter parameter) {
        return parameter.hasParameterAnnotation(this.annotationClass);
    }

    public abstract boolean isRequired(A annotation);

    @Nullable
    @Override
    public Object resolveArgument(
            @NonNull MethodParameter parameter,
            @NonNull DataFetchingEnvironment environment
    ) throws Exception {
        A annotation = parameter.getParameterAnnotation(this.annotationClass);
        Assert.state(annotation != null, "No " + annotationClass.getSimpleName() + " annotation");
        String name = parameter.getParameterName();
        if (name == null) {
            name = "";
        }
        MethodParameter nestedParameter = parameter.nestedIfOptional();
        Object arg = resolveValue(nestedParameter, environment);
        if (arg == null) {
            if (isRequired(annotation) && !nestedParameter.isOptional()) {
                handleMissingValue(parameter);
            }
            arg = handleNullValue(arg, nestedParameter.getNestedParameterType());
        }
        return arg;
    }

    protected abstract void handleMissingValue(MethodParameter parameter);

    @Nullable
    private Object handleNullValue(@Nullable Object value, Class<?> paramType) {
        if (value == null) {
            if (Boolean.TYPE.equals(paramType)) {
                return Boolean.FALSE;
            }
        }
        return value;
    }

    protected abstract @Nullable Object resolveValue(MethodParameter parameter, DataFetchingEnvironment environment);


}
