package org.beast.graphql.data.querydsl;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.querydsl.core.types.Predicate;
import lombok.Getter;
import org.beast.graphql.data.Edge;
import org.beast.graphql.data.PageInfo;
import org.beast.graphql.data.Pageable;
import org.springframework.lang.Nullable;

import java.util.List;
import java.util.stream.Collectors;

import static graphql.Assert.assertNotNull;

@Getter
public class PredicateConnection<T> {

    private Predicate predicate;

    private Pageable pageable;

    private final ImmutableList<T> nodes;
    private final ImmutableList<Edge<T>> edges;
    private final PageInfo pageInfo;

    public PredicateConnection(
            Predicate predicate,
            Pageable pageable,
            List<Edge<T>> edges
    ) {
        this(predicate, pageable, edges, null, null);
    }

    public PredicateConnection(
            Predicate predicate,
        Pageable pageable,
        List<Edge<T>> edges,
        @Nullable Boolean hasPreviousPage,
        @Nullable Boolean hasNextPage
    ) {
        assertNotNull(edges, () -> "edges cannot be null");
        this.predicate = predicate;
        this.pageable = pageable;
        this.nodes = ImmutableList.copyOf(edges.stream().map(Edge::getNode).collect(Collectors.toList()));
        this.edges = ImmutableList.copyOf(edges);
        this.pageInfo = buildPageInfo(
                Iterables.getFirst(edges, null),
                Iterables.getLast(edges, null),
                hasPreviousPage,
                hasNextPage
        );
    }

    public PageInfo buildPageInfo(
            Edge<T> start,
            Edge<T> end,
            Boolean hasPreviousPage,
            Boolean hasNextPage
    ) {
        return new PageInfo(
                start != null ? start.getCursor() : null,
                end != null ? end.getCursor() : null,
                hasPreviousPage,
                hasNextPage
        );
    }


}
