package org.beast.data.querydsl;

import com.google.common.collect.Lists;
import com.querydsl.core.BooleanBuilder;
import com.querydsl.core.types.ConstantImpl;
import com.querydsl.core.types.Ops;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.SimpleExpression;
import org.beast.data.domain.OrderCursor;
import org.beast.data.domain.RawCursor;
import org.bson.types.ObjectId;
import org.springframework.beans.BeanUtils;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.Property;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.data.annotation.Id;
import org.springframework.data.domain.Sort;
import org.springframework.data.mapping.PropertyPath;
import org.springframework.data.mapping.PropertyReferenceException;
import org.springframework.data.querydsl.EntityPathResolver;
import org.springframework.data.querydsl.binding.QuerydslPredicateBuilder;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.data.util.TypeInformation;
import org.springframework.lang.Nullable;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @see QuerydslPredicateBuilder
 *
 * //TODO 是否改造为ObjectStream 对象写入形式？
 */
public class CursorPredicateFactory {


    private final ConversionService conversionService;
    private final EntityPathResolver resolver;

    public CursorPredicateFactory(
            ConversionService conversionService,
            EntityPathResolver resolver,
            CursorEncoder encoder
    ) {
        this.conversionService = conversionService;
        this.resolver = resolver;
    }





    @Nullable
    public Predicate before(
            Class<?> clazz,
            Sort sort,
            RawCursor cursor
    ) {
        return this.predicate(clazz, sort, cursor, Direction.BEFORE);
    }

    @Nullable
    public Predicate after(
            Class<?> clazz,
            Sort sort,
            RawCursor cursor
    ) {
        return this.predicate(clazz, sort, cursor, Direction.AFTER);
    }
    private Predicate predicate(
            Class<?> clazz,
            Sort sort,
            RawCursor cursor,
            Direction direction
    ) {
        var typeInformation = ClassTypeInformation.from(clazz);
        return predicate(typeInformation, sort, cursor, direction);
    }

    @Nullable
    public <T> Predicate before(
            TypeInformation<T> typeInformation,
            Sort sort,
            RawCursor cursor
    ) {
        return this.predicate(typeInformation, sort, cursor, Direction.BEFORE);
    }
    @Nullable
    public <T> Predicate after(
            TypeInformation<T> typeInformation,
            Sort sort,
            RawCursor cursor
    ) {
        return this.predicate(typeInformation, sort, cursor, Direction.AFTER);
    }
    @Nullable
    public <T> Predicate after(
            Class<T> clazz,
            Sort sort,
            OrderCursor cursor
    ) {
        var typeInformation = ClassTypeInformation.from(clazz);
        return this.predicate(typeInformation, sort, cursor, Direction.AFTER);
    }


    private Predicate predicate(
            TypeInformation<?> typeInformation,
            Sort sort,
            RawCursor cursor,
            Direction direction
    ) {
        var rawValues = cursor.getOrders();
        List<Object> orderValues = new ArrayList<>(rawValues.size());
        int i = 0;
        for (Sort.Order order : sort) {
            var property = order.getProperty();
            PropertyPath path = getPropertyPath(property, typeInformation);
            if (path == null) {
                throw new IllegalArgumentException(
                        String.format("sort property: %s is miss", property)
                );
            }
            var entryValue = rawValues.get(i);
            Object value;
            try {
                value = convertToPropertyPathSpecificType(entryValue, path);
            } catch (Exception e) {
                throw new IllegalArgumentException(
                        String.format("'%s' does conversion failed", path),
                        e
                );
            }
            orderValues.add(value);
            i++;
        }
        var orderCursor = new OrderCursor(orderValues.toArray());
        return predicate(typeInformation, sort, orderCursor, direction);
    }
    @Nullable
    private Predicate predicate(
            TypeInformation<?> typeInformation,
            Sort sort,
            OrderCursor cursor,
            Direction direction
    ) {
        BooleanBuilder builder = new BooleanBuilder();
        if (cursor == null) {
            return builder.getValue();
        }
//        var typeInformation = ClassTypeInformation.from(clazz);
        BooleanBuilder basePredicate = new BooleanBuilder();

        int i = 0;
        for (Sort.Order order : sort) {
            var property = order.getProperty();
            Sort.Direction orderDirection = order.getDirection();
            PropertyPath path = getPropertyPath(property, typeInformation);
            if (path == null) {
                continue;
            }
            if (cursor.size() <= i) {
                throw new IllegalStateException(
                        String.format("'%s' does not appear to be a valid cursor", cursor)
                );
            }
            Object value = cursor.getCursorFor(i);

            //排序值筛选
            Optional<Predicate> orderPredicateOptional = invokeBinding(
                    path,
                    value,
                    ((direction == Direction.AFTER) == orderDirection.isDescending())? Ops.LT : Ops.GT
            );
            if (orderPredicateOptional.isPresent()) {
                builder.or(new BooleanBuilder(basePredicate).and(orderPredicateOptional.get()));
            }
            //排序值重复时，交给下一个排序项如(id),能确保不重复的游标筛选
            Optional<Predicate> eqPredicateOptional = invokeBinding(
                    path,
                    value,
                    Ops.EQ
            );
            if (eqPredicateOptional.isPresent()) {
                basePredicate = new BooleanBuilder(basePredicate).and(eqPredicateOptional.get());
            }
            i++;
        }
        return builder;
    }
    Optional<Predicate> invokeBinding(
            PropertyPath propertyPath,
            Object object,
            Ops ops
    ) {
        Path<?> path = reifyPath(resolver, propertyPath);
        if (path instanceof SimpleExpression expression) {
            if (object == null) {
                if (Ops.EQ == ops) {
                    return Optional.of(expression.isNull());
                }
                return Optional.empty();
            }
            return Optional.of(Expressions.booleanOperation(ops, expression, ConstantImpl.create(object)));
        }
        throw new IllegalArgumentException(
                String.format("Cannot create cursor predicate for path '%s' with type '%s'.", path, path.getMetadata().getPathType())
        );
    }

//    public ConnectionCursor getCursor(
//            TypeInformation<?> type,
//            Sort sort,
//            Object object) {
//        List<String> values = new ArrayList<>();
//        for (Sort.Order order : sort) {
//            String property = order.getProperty();
//            PropertyPath path = getPropertyPath(property, type);
//            if (path == null) {
//                continue;
//            }
//            String segment = path.getSegment();
//            TypeDescriptor source = getTargetTypeDescriptor(path);
//            PropertyDescriptor descriptor = BeanUtils.getPropertyDescriptor(type.getType(), segment);
//            if (descriptor != null) {
//                Method readMethod = descriptor.getReadMethod();
//                if (readMethod != null) {
//                    String value = null;
//                    try {
//                        if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
//                            readMethod.setAccessible(true);
//                        }
//                        Object entryValue = readMethod.invoke(object);
//                        value = (String) conversionService.convert(entryValue, source, TypeDescriptor.valueOf(String.class));
//
//                    } catch (Exception ex) {
//                        throw new FatalBeanException(
//                                "Could not create Cursor from property '" + readMethod.getName() + "'", ex);
//                    }
//                    values.add(value);
//                }
//            }
//
//
//        }
//        return new ConnectionCursor(values);
//    }
    private static Path<?> reifyPath(EntityPathResolver resolver, PropertyPath path) {
        Path<?> entityPath = resolver.createPath(path.getOwningType().getType());
        Field field = org.springframework.data.util.ReflectionUtils.findRequiredField(entityPath.getClass(),
                path.getSegment());
        Object value = ReflectionUtils.getField(field, entityPath);

        PropertyPath next = path.next();
        if (next != null) {
            return reifyPath(resolver, next);
        }
        return (Path<?>) value;
    }

    @Nullable
    PropertyPath getPropertyPath(String path, TypeInformation<?> type) {
        try {
            return PropertyPath.from(path, type);
        } catch (PropertyReferenceException e) {
            return null;
        }
    }
    private Object convertToPropertyPathSpecificType(String source, PropertyPath path) {
        Class<?> targetType = path.getLeafType();
        Field field = ReflectionUtils.findField(path.getOwningType().getType(), path.getSegment());
        if (field != null) {
            if (AnnotatedElementUtils.hasMetaAnnotationTypes(field, Id.class)) {
                targetType = ObjectId.class;
            };
        }
        if (!StringUtils.hasLength(source)) {
            return null;
        }
        return conversionService.canConvert(String.class, targetType) ?
            conversionService.convert(source, TypeDescriptor.forObject(source), getTargetTypeDescriptor(path)) : source;
    }

    private static TypeDescriptor getTargetTypeDescriptor(PropertyPath path) {

        Class<?> owningType = path.getLeafProperty().getOwningType().getType();
        String name = path.getLeafProperty().getSegment();
        PropertyDescriptor descriptor = BeanUtils.getPropertyDescriptor(
                owningType,
                name
        );

        TypeDescriptor result = descriptor == null //
                ? TypeDescriptor
                .nested(org.springframework.data.util.ReflectionUtils.findRequiredField(owningType, name), 0)
                : TypeDescriptor
                .nested(new Property(owningType, descriptor.getReadMethod(), descriptor.getWriteMethod(), name), 0);

        if (result == null) {
            throw new IllegalStateException(String.format("Could not obtain TypeDescriptor for PathInformation %s!", path));
        }

        return result;
    }



//    public Predicate before(@Nullable Cursor cursor, Sort sort) {
//
//    }


    public static enum Direction {
        AFTER,
        BEFORE,
    }
}
