package org.springframework.data.querydsl.binding;

import com.querydsl.core.BooleanBuilder;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.Predicate;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.Property;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.data.querydsl.EntityPathResolver;
import org.springframework.data.util.TypeInformation;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;

import java.beans.PropertyDescriptor;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;

public class CustomQuerydslPredicateBuilder extends QuerydslPredicateBuilder {

    private final MultiValueBinding<Path<? extends Object>, Object> defaultBinding;
    private final Map<PathInformation, Path<?>> paths;
    private final ConversionService conversionService;
    private final EntityPathResolver resolver;


    public CustomQuerydslPredicateBuilder(ConversionService conversionService, EntityPathResolver resolver) {
        super(conversionService, resolver);
        this.defaultBinding = new CustomQuerydslDefaultBinding();
        this.paths = new ConcurrentHashMap<>();
        this.conversionService = conversionService;
        this.resolver = resolver;
    }

    private static final Pattern PATTERN = Pattern.compile("^.*\\[.*]$");
    /**
     * 实现[]表达式处理
     */
    @Nullable
    public Predicate getPredicate(TypeInformation<?> type, MultiValueMap<String, String> values,
                                  QuerydslBindings bindings) {
        Assert.notNull(bindings, "Context must not be null!");

        BooleanBuilder builder = new BooleanBuilder();

        if (values.isEmpty()) {
            return builder.getValue();
        }

        for (Map.Entry<String, List<String>> entry : values.entrySet()) {

            if (isSingleElementCollectionWithoutText(entry.getValue())) {
                continue;
            }


            String path = entry.getKey();
            String property = path;
            String mapKey = null;
            if (path.contains("[") && PATTERN.matcher(path).matches() && path.endsWith("]")) {
                int propertySpitIndex = path.indexOf("[");
                property = property.substring(0, propertySpitIndex);
                mapKey = path.substring(propertySpitIndex + 1, path.length() - 1);
            }

            if (!bindings.isPathAvailable(property, type)) {
                continue;
            }

            PathInformation propertyPath = bindings.getPropertyPath(property, type);

            if (propertyPath == null) {
                continue;
            }

            //仅支持Map<String, String>
            Collection<Object> value = mapKey != null ? Collections.singleton(Map.of(mapKey, entry.getValue().get(0)))  : convertToPropertyPathSpecificType(entry.getValue(), propertyPath);
            Optional<Predicate> predicate = invokeBinding(propertyPath, bindings, value);

            predicate.ifPresent(builder::and);
        }

        return builder.getValue();
    }

    private static boolean isSingleElementCollectionWithoutText(List<String> source) {
        return source.size() == 1 && !StringUtils.hasLength(source.get(0));
    }

    private Optional<Predicate> invokeBinding(PathInformation dotPath, QuerydslBindings bindings,
                                              Collection<Object> values) {

        Path<?> path = getPath(dotPath, bindings);

        return bindings.getBindingForPath(dotPath).orElse(defaultBinding).bind(path, values);
    }

    private Path<?> getPath(PathInformation path, QuerydslBindings bindings) {

        Optional<Path<?>> resolvedPath = bindings.getExistingPath(path);

        return resolvedPath.orElseGet(() -> paths.computeIfAbsent(path, it -> it.reifyPath(resolver)));
    }
    private Collection<Object> convertToPropertyPathSpecificType(List<String> source, PathInformation path) {

        Class<?> targetType = path.getLeafType();

        if (source.isEmpty() || isSingleElementCollectionWithoutText(source)) {
            return Collections.emptyList();
        }

        Collection<Object> target = new ArrayList<>(source.size());

        for (String value : source) {

            target.add(conversionService.canConvert(String.class, targetType)
                    ? conversionService.convert(value, TypeDescriptor.forObject(value), getTargetTypeDescriptor(path))
                    : value);
        }

        return target;
    }

    private static TypeDescriptor getTargetTypeDescriptor(PathInformation path) {

        PropertyDescriptor descriptor = path.getLeafPropertyDescriptor();

        Class<?> owningType = path.getLeafParentType();
        String leafProperty = path.getLeafProperty();

        TypeDescriptor result = descriptor == null //
                ? TypeDescriptor
                .nested(org.springframework.data.util.ReflectionUtils.findRequiredField(owningType, leafProperty), 0)
                : TypeDescriptor
                .nested(new Property(owningType, descriptor.getReadMethod(), descriptor.getWriteMethod(), leafProperty), 0);

        if (result == null) {
            throw new IllegalStateException(String.format("Could not obtain TypeDesciptor for PathInformation %s!", path));
        }

        return result;
    }
}
