package org.beast.data.jackson;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import org.beast.data.jackson.relay.RelayJacksonModule;
import org.beast.data.jackson.ser.*;
import org.beast.data.message.ErrorMessageAutoConfiguration;
import org.beast.data.message.ErrorMessageSource;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.autoconfigure.jackson.JacksonAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.data.domain.Page;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

@Configuration(proxyBeanMethods = false)
@ConditionalOnClass(ObjectMapper.class)
@AutoConfigureBefore(JacksonAutoConfiguration.class)
@AutoConfigureAfter(ErrorMessageAutoConfiguration.class)
public class JacksonCustomAutoConfiguration {

    @Configuration
    @ConditionalOnClass(Jackson2ObjectMapperBuilder.class)
    static class JacksonCustomizerConfiguration {

        @Order(1)
        @Bean
        public Jackson2ObjectMapperBuilderCustomizer defaultJackson2ObjectMapperBuilderCustomizer() {
            return jacksonObjectMapperBuilder -> {
                jacksonObjectMapperBuilder.serializationInclusion(JsonInclude.Include.NON_NULL);
                jacksonObjectMapperBuilder.featuresToEnable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
                jacksonObjectMapperBuilder.featuresToDisable(SerializationFeature.WRITE_DATE_TIMESTAMPS_AS_NANOSECONDS);
                jacksonObjectMapperBuilder.featuresToDisable(DeserializationFeature.READ_DATE_TIMESTAMPS_AS_NANOSECONDS);
                //全局设置 避免客户端精度不足(如:js) 解析出现精度问题导致出BUG， (如userId), 反序列化是支持得
                jacksonObjectMapperBuilder.serializerByType(Long.class, ToStringSerializer.instance);
                jacksonObjectMapperBuilder.serializerByType(Long.TYPE, ToStringSerializer.instance);
            };
        }


        @Order(1)
        @ConditionalOnClass(Page.class)
        @Bean
        public Jackson2ObjectMapperBuilderCustomizer defaultPageJackson2ObjectMapperBuilderCustomizer() {
            return jacksonObjectMapperBuilder -> jacksonObjectMapperBuilder.serializers(new PageSerializer());
        }


        @Bean
        public RelayJacksonModule relayJacksonModule() {
            return new RelayJacksonModule();
        }


        @Bean
        @ConditionalOnBean(ErrorMessageSource.class)
        public ErrorModule errorModule(ErrorMessageSource source) {
            return new ErrorModule(source);
        }

        @Bean
        @ConditionalOnClass(MongoOperations.class)
        public MongoModule mongoModule() {
            return new MongoModule();
        }

        @Bean
        public MoneyModule moneyModule() {
            return new MoneyModule();
        }


    }
}
