package org.beast.data.web;

import org.beast.data.relay.Cursor;
import org.beast.data.relay.Pageable;
import org.beast.data.relay.PageableRequest;
import org.springframework.core.MethodParameter;
import org.springframework.data.domain.Sort;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;

import javax.validation.constraints.Null;
import java.util.Objects;
import java.util.Optional;

public class PageableHandlerMethodArgumentResolverSupport {

    private static final String DEFAULT_FIRST_PARAMETER = "first";
    private static final String DEFAULT_OFFSET_PARAMETER = "offset";
    private static final String DEFAULT_SORT_PARAMETER = "sort";
    private static final String DEFAULT_DIRECTION_PARAMETER = "direction";
    private Pageable fallbackPageable = PageableRequest.of(null, null, 10, null, 0);
    private boolean oneIndexedParameters = false;


    protected int maxOffset = 2000 * 100;

    protected Pageable getPageable(
           MethodParameter methodParameter,
           @Nullable Cursor after,
           @Nullable Cursor before,
           @Nullable Integer first,
           @Nullable Integer offset
    ) {

        var defaultOrFallback = Optional.ofNullable(getDefaultFromAnnotationOrFallback(methodParameter));

        //判断是否分页


        first = Optional.ofNullable(first).orElseGet(() -> defaultOrFallback.map(Pageable::getFirst).orElseThrow());
        offset = Optional.ofNullable(offset).orElseGet(() -> defaultOrFallback.map(Pageable::getOffset).orElseThrow());



        return PageableRequest.of(
                after, before,
                first, null, offset,
                defaultOrFallback.map(Pageable::getSort)
                        .orElseGet(Sort::unsorted)
        );
    }

    private Optional<Integer> parseAndApplyBoundaries(@Nullable String parameter, int upper, boolean shiftIndex) {

        if (!StringUtils.hasText(parameter)) {
            return Optional.empty();
        }

        try {
            int parsed = Integer.parseInt(parameter) - (oneIndexedParameters && shiftIndex ? 1 : 0);
            return Optional.of(parsed < 0 ? 0 : parsed > upper ? upper : parsed);
        } catch (NumberFormatException e) {
            return Optional.of(0);
        }
    }

    private Pageable getDefaultFromAnnotationOrFallback(MethodParameter methodParameter) {

        PageableDefault defaults = methodParameter.getParameterAnnotation(PageableDefault.class);

        if (defaults != null) {
            return getDefaultPageRequestFrom(methodParameter, defaults);
        }

        return fallbackPageable;
    }

    private static Pageable getDefaultPageRequestFrom(MethodParameter parameter, PageableDefault defaults) {
        var first = defaults.first();
        var offset = defaults.offset();
        var sort = Sort.by(defaults.direction(), defaults.sort());
        return PageableRequest.of(
                null, null, first, null, offset, sort
        );
    }
}
