package org.beast.data.web;

import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.beast.data.relay.SimpleCursor;
import org.beast.data.relay.Pageable;
import org.springframework.core.MethodParameter;
import org.springframework.data.web.SortArgumentResolver;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;

import java.util.Optional;

@Slf4j
public class PageableHandlerMethodArgumentResolver extends PageableHandlerMethodArgumentResolverSupport implements HandlerMethodArgumentResolver {

    private SortArgumentResolver sortResolver;

    private static final int DEFAULT_MAX_PAGE_SIZE = 2000;

    private boolean oneIndexedParameters = false;

    public PageableHandlerMethodArgumentResolver(SortArgumentResolver sortResolver) {
        this.sortResolver = sortResolver;
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return Pageable.class.equals(parameter.getParameterType());
    }

    @Nullable
    @Override
    public Object resolveArgument(MethodParameter parameter,
                                  @Nullable ModelAndViewContainer mavContainer,
                                  NativeWebRequest webRequest, @Nullable WebDataBinderFactory binderFactory) throws Exception {
        var after = webRequest.getParameter("after");
        var first = webRequest.getParameter("first");
        var offset = webRequest.getParameter("offset");
        String[] directionParameter = webRequest.getParameterValues("sort");

        var pageable = getPageable(parameter, after, first, offset);

        var sort = sortResolver.resolveArgument(parameter, mavContainer, webRequest, binderFactory);

        if (sort.isSorted()) {
            return pageable.withSort(sort);
        }
        return pageable;
    }

    private Pageable getPageable(MethodParameter parameter, String afterString, String firstString, String offsetString) {
        var after = SimpleCursor.ofNullable(afterString);
        var first = parseAndApplyBoundaries(firstString, Integer.MAX_VALUE, true);
        var offset = parseAndApplyBoundaries(offsetString, this.maxOffset, true);


        return getPageable(parameter, after, null, first.orElse(null), offset.orElse(null));
    }

    private Optional<Integer> parseAndApplyBoundaries(@Nullable String parameter, int upper, boolean shiftIndex) {

        if (!StringUtils.hasText(parameter)) {
            return Optional.empty();
        }

        try {
            int parsed = Integer.parseInt(parameter) - (oneIndexedParameters && shiftIndex ? 1 : 0);
            return Optional.of(parsed < 0 ? 0 : parsed > upper ? upper : parsed);
        } catch (NumberFormatException e) {
            return Optional.of(0);
        }
    }
}

