package org.beast.data.message;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.io.Serializable;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

@ToString
@Getter @Setter
public class ReturnResult<T> implements Serializable {

    @Nullable
    private T data;

    @Nullable
    private ErrorInfo error;



    public ReturnResult() {
    }

    public ReturnResult(@Nullable T data, @Nullable ErrorInfo error) {
        this.data = data;
        this.error = error;
    }

    public boolean isError() {
        return this.error != null;
    }
    public boolean hasError() {
        return this.error != null;
    }

    public boolean isSuccess() {
        return this.error == null;
    }

    public boolean isPresent() {
        return this.data != null;
    }

    public boolean isError(Predicate<ErrorInfo> predicate) {
        if (hasError()) {
            return predicate.test(error);
        }
        return false;
    }
    public boolean isError(String code) {
        return isError((error) -> {
            return error.is(code);
        });
    }
    public void ifPresent(Consumer<? super T> action) {
        if (data != null) {
            action.accept(data);
        }
    }

    public boolean isEmpty() {
        return Objects.isNull(this.data) && Objects.isNull(this.error);
    }
    public boolean hasData() {
        return Objects.nonNull(this.data);
    }


    @NonNull
    public T get() {
        if (data == null) {
            throw new NoSuchElementException("No value present");
        }
        return data;
    }

    public T orElse(T other) {
        if (hasError()) {
            return other;
        }
        return data;
    }

    public T orElseGet(Supplier<? extends T> supplier) {
        if (hasError()) {
            return supplier.get();
        }
        return data;
    }

    public T orElseThrow() {
        if (hasError()) {
            assert error != null;
            throw new ErrorException(error);
        }
        return data;
    }

    public <X extends Throwable> T orElseThrow(ErrorSupplier<X, ErrorInfo> exceptionSupplier) throws X {
        if (hasError()) {
            throw exceptionSupplier.get(getError());
        }
        return data;
    }

    public <X extends Throwable> T orElseThrow(Supplier<? extends X> exceptionSupplier) throws X {
        if (hasError()) {
            throw exceptionSupplier.get();
        }
        return data;
    }

    public Optional<T> optionalOrThrow(MessageErrorOwner owner) {
        return optionalOrThrow(owner.getErrorCode());
    }
    public Optional<T> optionalOrThrow(String emptyCode) {
        return optionalOrThrow(error -> {
            return error.is(emptyCode);
        });
    }
    public Optional<T> optionalOrThrow(Predicate<ErrorInfo> emptyPredicate) {
        if (hasError()) {
            if (emptyPredicate.test(error)) {
                return Optional.empty();
            }
            assert error != null;
            throw new ErrorException(error);
        }
        return Optional.ofNullable(data);
    }

    public ReturnResult<T> filter(@NonNull Predicate<T> predicate) {
        if (!isPresent()) {
            return this;
        } else {
            return new ReturnResult<>(predicate.test(this.data) ? this.data : null, error);
        }
    }
    public <O> ReturnResult<O> map(Function<T, O> converter) {
        if (hasError()) {
            return new ReturnResult<>(null, error);
        }
        return new ReturnResult<>(converter.apply(this.data), error);
    }

}
