package org.beast.data.openfeign.support;

import feign.RequestTemplate;
import feign.codec.EncodeException;
import feign.codec.Encoder;
import lombok.extern.slf4j.Slf4j;
import org.beast.data.relay.Pageable;
import org.springframework.data.domain.Sort;

import java.lang.reflect.Type;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;

@Slf4j
public class PageableSpringEncoder implements Encoder {

    private final Encoder delegate;

    private String beforeParameter = "before";

    private String afterParameter = "after";

    private String firstParameter = "first";

    private String sortParameter = "sort";

    private String offsetParameter = "offset";

    public PageableSpringEncoder(Encoder delegate) {
        this.delegate = delegate;
    }

    @Override
    public void encode(Object object, Type bodyType, RequestTemplate template) throws EncodeException {
        if (supports(object)) {
            Pageable pageable = (Pageable) object;
            query(template, firstParameter, pageable.getFirst());
            query(template, beforeParameter, pageable.getBefore());
            query(template, afterParameter, pageable.getAfter());
            query(template, offsetParameter, pageable.getOffset());
            if (Objects.nonNull(pageable.getSort())) {
                applySort(template, pageable.getSort());
            }
        } else {
            if (delegate != null) {
                delegate.encode(object, bodyType, template);
            }
        }
    }
    private void query(RequestTemplate template, String name, Object value) {
        if (Objects.nonNull(value)) {
            var encodedValue = URLEncoder.encode(String.valueOf(value), StandardCharsets.UTF_8);
            template.query(name, encodedValue);
        }

    }



    private void applySort(RequestTemplate template, Sort sort) {
        Collection<String> existingSorts = template.queries().get("sort");
        List<String> sortQueries = existingSorts != null ? new ArrayList<>(existingSorts) : new ArrayList<>();
        if (!sortParameter.equals("sort")) {
            existingSorts = template.queries().get(sortParameter);
            if (existingSorts != null) {
                sortQueries.addAll(existingSorts);
            }
        }
        for (Sort.Order order : sort) {
            sortQueries.add(order.getProperty() + "%2C" + order.getDirection());
        }
        if (!sortQueries.isEmpty()) {
            template.query(sortParameter, sortQueries);
        }
    }

    protected boolean supports(Object object) {
        return object instanceof Pageable;
    }


}
