package org.beast.data.util;

import org.beast.data.relay.Cursor;
import org.beast.data.relay.DefaultCursor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.FatalBeanException;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.Property;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.data.domain.Sort;
import org.springframework.data.mapping.PropertyPath;
import org.springframework.data.mapping.PropertyReferenceException;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.data.util.TypeInformation;
import org.springframework.lang.Nullable;
import org.springframework.util.ClassUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

public class CursorEncoder {

    private final ConversionService conversionService;

    public CursorEncoder(
            ConversionService conversionService
    ) {
        this.conversionService = conversionService;
    }

    public Cursor encode(
            Sort sort,
            Object object
    ) {
        return this.encode(sort, object, ClassUtils.getUserClass(object));
    }
    public Cursor encode(
            Sort sort,
            Object object,
            Class<?> clazz
    ) {
        return this.encode(ClassTypeInformation.from(clazz), sort, object);
    }
    public Cursor encode(
            TypeInformation<?> type,
            Sort sort,
            Object object) {
        List<String> values = new ArrayList<>();
        for (Sort.Order order : sort) {
            String property = order.getProperty();
            PropertyPath path = getPropertyPath(property, type);
            if (path == null) {
                continue;
            }
            String segment = path.getSegment();
            TypeDescriptor source = getTargetTypeDescriptor(path);
            PropertyDescriptor descriptor = BeanUtils.getPropertyDescriptor(type.getType(), segment);
            if (descriptor != null) {
                Method readMethod = descriptor.getReadMethod();
                if (readMethod != null) {
                    String value = null;
                    try {
                        if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                            readMethod.setAccessible(true);
                        }
                        Object entryValue = readMethod.invoke(object);
                        value = (String) conversionService.convert(entryValue, source, TypeDescriptor.valueOf(String.class));

                    } catch (Exception ex) {
                        throw new FatalBeanException(
                                "Could not create Cursor from property '" + readMethod.getName() + "'", ex);
                    }
                    values.add(value);
                }
            }
        }
        return DefaultCursor.valueOfCursors(values);
    }

//    public List<String> decodeProperty(
//            TypeInformation<?> type,
//            Sort sort,
//            ConnectionCursor cursor
//    ) {
//        List<String> values = new ArrayList<>();
//        for (Sort.Order order : sort) {
//            String property = order.getProperty();
//            PropertyPath path = getPropertyPath(property, type);
//            if (path == null) {
//                continue;
//            }
//            String segment = path.getSegment();
//            TypeDescriptor source = getTargetTypeDescriptor(path);
//            PropertyDescriptor descriptor = BeanUtils.getPropertyDescriptor(type.getType(), segment);
//            if (descriptor != null) {
//                Method readMethod = descriptor.getReadMethod();
//                if (readMethod != null) {
//                    String value = null;
//                    try {
//                        if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
//                            readMethod.setAccessible(true);
//                        }
//                        Object entryValue = readMethod.invoke(object);
//                        value = (String) conversionService.convert(entryValue, source, TypeDescriptor.valueOf(String.class));
//
//                    } catch (Exception ex) {
//                        throw new FatalBeanException(
//                                "Could not create Cursor from property '" + readMethod.getName() + "'", ex);
//                    }
//                    values.add(value);
//                }
//            }
//        }
//    }

    @Nullable
    PropertyPath getPropertyPath(String path, TypeInformation<?> type) {
        try {
            return PropertyPath.from(path, type);
        } catch (PropertyReferenceException e) {
            return null;
        }
    }

    private static TypeDescriptor getTargetTypeDescriptor(PropertyPath path) {

        Class<?> owningType = path.getLeafProperty().getOwningType().getType();
        String name = path.getLeafProperty().getSegment();
        PropertyDescriptor descriptor = BeanUtils.getPropertyDescriptor(
                owningType,
                name
        );

        TypeDescriptor result = descriptor == null //
                ? TypeDescriptor
                .nested(org.springframework.data.util.ReflectionUtils.findRequiredField(owningType, name), 0)
                : TypeDescriptor
                .nested(new Property(owningType, descriptor.getReadMethod(), descriptor.getWriteMethod(), name), 0);

        if (result == null) {
            throw new IllegalStateException(String.format("Could not obtain TypeDesciptor for PathInformation %s!", path));
        }

        return result;
    }


}
