package org.beast.data.function;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.FatalBeanException;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.data.util.TypeInformation;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;

public class BeanFieldsCopyFunction<I, O> {

    protected final Class<O> entityClass;
    public BeanFieldsCopyFunction() {
        this(null);
    }


    public BeanFieldsCopyFunction(Class<O> clazz) {
        if (clazz == null) {
            Class<?> copyFunction = this.getClass();
            List<TypeInformation<?>> arguments = ClassTypeInformation.from(copyFunction)
                    .getRequiredSuperTypeInformation(BeanFieldsCopyFunction.class)
                    .getTypeArguments();
            clazz = (Class<O>) resolveTypeParameter(arguments, 1,
                    () -> String.format("Could not resolve Output type of %s!", copyFunction));
        }
        this.entityClass = clazz;
    }
    private static Class<?> resolveTypeParameter(List<TypeInformation<?>> arguments, int index,
                                                 Supplier<String> exceptionMessage) {

        if (arguments.size() <= index || arguments.get(index) == null) {
            throw new IllegalArgumentException(exceptionMessage.get());
        }

        return arguments.get(index).getType();
    }

    private @NonNull
    O newOutputInstance()  {
        return BeanUtils.instantiateClass(this.entityClass);
    }


    public Function<I, O> newFunction(String ...fields) {
        return newFunction(Arrays.asList(fields));
    }

    public Function<I, O> newFunction(List<String> fields) {
        return new Function<I, O>() {
            @Override
            public O apply(I input) {
                O output = BeanFieldsCopyFunction.this.newOutputInstance();
                Assert.notNull(output, "BeanFieldsCopyFunction O Annotation require not null");
                copyProperties(input, output, fields);
                return output;
            }
        };
    }
    private static void copyProperties(Object source, Object target, List<String> fields) throws BeansException {

        Assert.notNull(source, "Source must not be null");
        Assert.notNull(target, "Target must not be null");

        Class<?> actualEditable = target.getClass();
        PropertyDescriptor[] targetPds = BeanUtils.getPropertyDescriptors(actualEditable);


        for (PropertyDescriptor targetPd : targetPds) {
            Method writeMethod = targetPd.getWriteMethod();
            if (writeMethod != null && (fields == null || fields.contains(targetPd.getName()))) {
                PropertyDescriptor sourcePd = BeanUtils.getPropertyDescriptor(source.getClass(), targetPd.getName());
                if (sourcePd != null) {
                    Method readMethod = sourcePd.getReadMethod();
                    if (readMethod != null &&
                            ClassUtils.isAssignable(writeMethod.getParameterTypes()[0], readMethod.getReturnType())) {
                        try {
                            if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                                readMethod.setAccessible(true);
                            }
                            Object value = readMethod.invoke(source);
                            if (!Modifier.isPublic(writeMethod.getDeclaringClass().getModifiers())) {
                                writeMethod.setAccessible(true);
                            }
                            writeMethod.invoke(target, value);
                        }
                        catch (Throwable ex) {
                            throw new FatalBeanException(
                                    "Could not copy property '" + targetPd.getName() + "' from source to target", ex);
                        }
                    }
                }
            }
        }
    }
}
