package org.beast.data.domain;

import org.javamoney.moneta.spi.MoneyUtils;

import javax.annotation.Nonnull;
import java.math.BigDecimal;
import java.util.Objects;

/**
 * 简单金额
 *
 */
public class SimpleMoney implements Comparable<SimpleMoney>{

    private final long value;
    public static final int SCALE = 2;


    public static final SimpleMoney MAX_VALUE = new SimpleMoney(1000000000000L);
    private static final BigDecimal MAX_BD = MAX_VALUE.getBigDecimal();
    public static final SimpleMoney MIN_VALUE = new SimpleMoney(-1000000000000L);
    private static final BigDecimal MIN_BD = MIN_VALUE.getBigDecimal();


    private SimpleMoney(Number value) {
        Objects.requireNonNull(value, "Number is required.");
        this.value = this.getInternalNumber(value);
    }

    private SimpleMoney(long value) {
        this.value = value;
    }

    private long getInternalNumber(Number number) {
        BigDecimal bd = MoneyUtils.getBigDecimal(number);
        if (bd.scale() > SCALE) {
            throw new ArithmeticException(number + " can not be represented by this class, scale > " + 5);
        } else if (bd.compareTo(MIN_BD) < 0) {
            throw new ArithmeticException("Overflow: " + number + " < " + MIN_BD);
        } else if (bd.compareTo(MAX_BD) > 0) {
            throw new ArithmeticException("Overflow: " + number + " > " + MAX_BD);
        } else {
            return bd.movePointRight(SCALE).longValue();
        }
    }

    public static SimpleMoney of(Number value) {
        return new SimpleMoney(value);
    }

    public static SimpleMoney of(String value) {
        return new SimpleMoney(new BigDecimal(value));
    }

    public static SimpleMoney ofNullable(Number value) {
        return value == null ? null : of(value);
    }

    public static SimpleMoney ofNullable(String value) {
        return value == null ? null : of(value);
    }

    @Override
    public int compareTo(@Nonnull SimpleMoney o) {
        Objects.requireNonNull(o);
        return this.getBigDecimal().compareTo(o.getBigDecimal());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SimpleMoney that = (SimpleMoney) o;
        return value == that.value;
    }



    public BigDecimal getBigDecimal() {
        return BigDecimal.valueOf(this.value).movePointLeft(2);
    }


    public boolean isZero() {
        return this.value == 0L;
    }

    public boolean isPositive() {
        return this.value > 0L;
    }

    public boolean isPositiveOrZero(){
        return this.value >= 0L;
    }

    public boolean isNegative(){
        return this.value < 0L;
    }

    public boolean isNegativeOrZero(){
        return this.value <= 0L;
    }

    /**
     * 范围
     */
    public int scale(){
        return SimpleMoney.SCALE;
    }

    /**
     * 精度
     */
    public int precision() {
        return getBigDecimal().precision();
    }


    public int signum() {
        if (this.value < 0) {
            return -1;
        }
        if (this.value == 0) {
            return 0;
        }
        return 1;
    }

    public boolean isLessThan() {
        throw new UnsupportedOperationException();
    }
    public boolean isLessThanOrEqualTo() {
        throw new UnsupportedOperationException();
    }

    public boolean isGreaterThan() {
        throw new UnsupportedOperationException();
    }
    public boolean isGreaterThanOrEqualTo() {
        throw new UnsupportedOperationException();
    }

    public boolean isEqualTo(SimpleMoney amount) {
        return this.value == amount.value;
    }

    @Override
    public String toString() {
        return this.getBigDecimal().toString();
    }
}

