package org.beast.data.domain;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import org.springframework.lang.NonNull;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Objects;

public class OrdersCursor implements Cursor, Comparable<OrdersCursor> {

    //createdAt,id
    //如createdAt:desc, id:asc, 先按createAt排序 如有重复值按id排序
    private final List<String> orders;

    private final String rawValue;
    public OrdersCursor(List<String> orders) {
        this.orders = orders;
        this.rawValue = getRawValueFromOrders(orders);
    }

    public OrdersCursor(String rawValue) {
        this.rawValue = rawValue;
        this.orders = getOrdersFromRawValues(rawValue);
    }


    public static String getRawValueFromOrders(List<String> orders) {
        List<String> encodeValues = new ArrayList<>();
        for (String value : orders) {
            if (Objects.isNull(value)) {
                value = "";
            }
            String encodedValue = Base64.getEncoder().encodeToString(value.getBytes(StandardCharsets.UTF_8));
            encodeValues.add(encodedValue);

        }
        return Joiner.on(".").join(encodeValues);
    }

    public static List<String> getOrdersFromRawValues(String rawValue) {
        List<String> values = new ArrayList<>();
        Base64.Decoder decoder = Base64.getDecoder();
        for (String encodeValue : Splitter.on(".").split(rawValue)) {
            values.add(new String(decoder.decode(encodeValue), StandardCharsets.UTF_8));
        }
        return values;
    }

    @JsonCreator
    public static OrdersCursor ofNullable(String rawValue) {
        if (rawValue == null) {
            return null;
        }
        return new OrdersCursor(rawValue);
    }

    public List<String> getOrders() {
        return this.orders;
    }

    @Override
    public String getValue() {
        return this.rawValue;
    }

    @Override
    public String toString() {
        return this.getValue();
    }

    @Override
    public int compareTo(@NonNull OrdersCursor o) {
        return getValue().compareTo(o.getValue());
    }
}
