package org.beast.data.converter;

import org.beast.data.domain.SimpleMoney;
import org.springframework.core.convert.converter.Converter;
import org.springframework.data.convert.ReadingConverter;
import org.springframework.data.convert.WritingConverter;
import org.springframework.lang.NonNull;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

public class MoneyConverters {

    public static List<?> getConvertersToRegister() {
        List<Object> converters = new ArrayList<>();
        converters.add(SimpleMoneyToLongConvert.INSTANCE);
        converters.add(LongToSimpleMoneyConvert.INSTANCE);
        converters.add(IntegerToSimpleMoneyConvert.INSTANCE);
        converters.add(StringToSimpleMoneyConvert.INSTANCE);
        return converters;
    }


    @WritingConverter
    public enum SimpleMoneyToLongConvert implements Converter<SimpleMoney, Long> {
        INSTANCE;
        @Override
        public Long convert(@NonNull SimpleMoney source) {
            BigDecimal value = source.getBigDecimal().movePointRight(SimpleMoney.SCALE);
            if (value.scale() != 0) {
                throw new ArithmeticException(source + " can not be represented by this class, scale > " + SimpleMoney.SCALE);
            }
            return value.longValue();
        }
    }

    @ReadingConverter
    public enum LongToSimpleMoneyConvert implements Converter<Long, SimpleMoney> {
        INSTANCE;
        @Override
        public SimpleMoney convert(@NonNull Long source) {
            BigDecimal value = new BigDecimal(source).movePointLeft(SimpleMoney.SCALE);
            return SimpleMoney.of(value);
        }
    }

    @ReadingConverter
    public enum IntegerToSimpleMoneyConvert implements Converter<Integer, SimpleMoney> {
        INSTANCE;
        @Override
        public SimpleMoney convert(@NonNull Integer source) {
            BigDecimal value = new BigDecimal(source).movePointLeft(SimpleMoney.SCALE);
            return SimpleMoney.of(value);
        }
    }

    @ReadingConverter
    public enum StringToSimpleMoneyConvert implements Converter<String, SimpleMoney> {
        INSTANCE;
        @Override
        public SimpleMoney convert(@NonNull String source) {
            BigDecimal value = new BigDecimal(source);
            return SimpleMoney.of(value);
        }
    }
}
