package org.beast.data.message;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

import java.io.Serializable;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

@ToString
@Getter @Setter
public class ReturnResult<T> implements Serializable {

    private T data;

    private ErrorInfo error;



    public ReturnResult() {
    }

    public ReturnResult(T data, ErrorInfo error) {
        this.data = data;
        this.error = error;
    }

    public boolean hasError() {
        return this.error != null;
    }

    public boolean isSuccess() {
        return this.error == null;
    }

    public boolean isPresent() {
        return this.data != null;
    }
    public void ifPresent(Consumer<? super T> action) {
        if (data != null) {
            action.accept(data);
        }
    }

    public T orElse(T other) {
        if (hasError()) {
            return other;
        }
        return data;
    }

    public T orElseGet(Supplier<? extends T> supplier) {
        if (hasError()) {
            return supplier.get();
        }
        return data;
    }

    public T orElseThrow() {
        if (hasError()) {
            throw new ErrorException(error);
        }
        return data;
    }

    public <X extends Throwable> T orElseThrow(ErrorSupplier<X, ErrorInfo> exceptionSupplier) throws X {
        if (hasError()) {
            throw exceptionSupplier.get(getError());
        }
        return data;
    }

    public <X extends Throwable> T orElseThrow(Supplier<? extends X> exceptionSupplier) throws X {
        if (hasError()) {
            throw exceptionSupplier.get();
        }
        return data;
    }

    public <O> ReturnResult<O> map(Function<T, O> converter) {
        return new ReturnResult<>(converter.apply(this.data), error);
    }


}
