/**
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.apache.geronimo.gbean;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.geronimo.kernel.management.NotificationType;

/**
 * Describes a GBean.  This class should never be constructed directly.  Insted use GBeanInfoBuilder.
 *
 * @version $Rev: 487175 $ $Date: 2006-12-14 03:10:31 -0800 (Thu, 14 Dec 2006) $
 */
public final class GBeanInfo implements Serializable {
    private static final long serialVersionUID = -6198804067155550221L;
    
    public static final int PRIORITY_CLASSLOADER = 1;
    public static final int PRIORITY_NORMAL = 5;

    private static final Set DEFAULT_NOTIFICATIONS = Collections.unmodifiableSet(new HashSet(Arrays.asList(NotificationType.TYPES)));

    /**
     * Static helper to try to get the GBeanInfo from the class supplied.
     *
     * @param className   name of the class to get the GBeanInfo from
     * @param classLoader the class loader use to load the specifiec class
     * @return GBeanInfo generated by supplied class
     * @throws InvalidConfigurationException
     *          if there is a problem getting the GBeanInfo from the class
     */
    public static GBeanInfo getGBeanInfo(String className, ClassLoader classLoader) throws InvalidConfigurationException {
        Class clazz;
        try {
            clazz = classLoader.loadClass(className);
        } catch (ClassNotFoundException e) {
            throw new InvalidConfigurationException("Could not load class " + className, e);
        } catch (NoClassDefFoundError e) {
            throw new InvalidConfigurationException("Could not load class " + className, e);
        }
        Method method;
        try {
            method = clazz.getDeclaredMethod("getGBeanInfo", new Class[]{});
        } catch (NoSuchMethodException e) {
            try {
                // try to get the info from ${className}GBean
                clazz = classLoader.loadClass(className + "GBean");
                method = clazz.getDeclaredMethod("getGBeanInfo", new Class[]{});
            } catch (Exception ignored) {
                throw new InvalidConfigurationException("Class does not have a getGBeanInfo() method: " + className);
            }
        } catch (NoClassDefFoundError e) {
            throw new InvalidConfigurationException("Could not find getGBeanInfo method on " + className, e);
        }
        try {
            return (GBeanInfo) method.invoke(null, new Object[]{});
        } catch (Exception e) {
            throw new InvalidConfigurationException("Could not get GBeanInfo from class: " + className, e);
        }
    }

    private final String sourceClass;
    private final String name;
    private final String className;
    private final String j2eeType;
    private final Set attributes;
    private final Map attributesByName;
    private final GConstructorInfo constructor;
    private final Set operations;
    private final Set notifications;
    private final Set references;
    private final Map referencesByName;
    private final Set interfaces;
    private int priority;

    /**
     * @deprecated use GBeanInfoBuilder
     */
    public GBeanInfo(String name, String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces) {
        this(null, name, className, j2eeType, attributes, constructor, operations, references, interfaces, DEFAULT_NOTIFICATIONS, PRIORITY_NORMAL);
    }

    /**
     * @deprecated use GBeanInfoBuilder
     */
    public GBeanInfo(String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces) {
        this(null, className, className, j2eeType, attributes, constructor, operations, references, interfaces, DEFAULT_NOTIFICATIONS, PRIORITY_NORMAL);
    }

    /**
     * @deprecated use GBeanInfoBuilder
     */
    public GBeanInfo(String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces, Set notifications) {
        this(null, className, className, j2eeType, attributes, constructor, operations, references, interfaces, notifications, PRIORITY_NORMAL);
    }

    /**
     * @deprecated use GBeanInfoBuilder
     */
    public GBeanInfo(String name, String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces, Set notifications) {
        this(null, name, className, j2eeType, attributes, constructor, operations, references, interfaces, notifications, PRIORITY_NORMAL);
    }

    GBeanInfo(String sourceClass, String name, String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces, int priority) {
        this(sourceClass, name, className, j2eeType, attributes, constructor, operations, references, interfaces, DEFAULT_NOTIFICATIONS, priority);
    }

    GBeanInfo(String sourceClass, String name, String className, String j2eeType, Collection attributes, GConstructorInfo constructor, Collection operations, Set references, Set interfaces, Set notifications, int priority) {
        this.sourceClass = sourceClass;
        this.name = name;
        this.className = className;
        this.j2eeType = j2eeType;
        if (attributes == null) {
            this.attributes = Collections.EMPTY_SET;
            this.attributesByName = Collections.EMPTY_MAP;
        } else {
            Map map = new HashMap();
            for (Iterator iterator = attributes.iterator(); iterator.hasNext();) {
                GAttributeInfo attribute = (GAttributeInfo) iterator.next();
                map.put(attribute.getName(), attribute);
            }
            this.attributesByName = Collections.unmodifiableMap(map);
            this.attributes = Collections.unmodifiableSet(new HashSet(map.values()));
        }
        if (constructor == null) {
            this.constructor = new GConstructorInfo(Collections.EMPTY_LIST);
        } else {
            this.constructor = constructor;
        }
        if (operations == null) {
            this.operations = Collections.EMPTY_SET;
        } else {
            this.operations = Collections.unmodifiableSet(new HashSet(operations));
        }
        if (references == null) {
            this.references = Collections.EMPTY_SET;
            this.referencesByName = Collections.EMPTY_MAP;
        } else {
            Map map = new HashMap();
            for (Iterator iterator = references.iterator(); iterator.hasNext();) {
                GReferenceInfo reference = (GReferenceInfo) iterator.next();
                map.put(reference.getName(), reference);
            }
            this.referencesByName = Collections.unmodifiableMap(map);
            this.references = Collections.unmodifiableSet(new HashSet(references));
        }
        if (interfaces == null) {
            this.interfaces = Collections.EMPTY_SET;
        } else {
            this.interfaces = Collections.unmodifiableSet(new HashSet(interfaces));
        }
        if (notifications == null) {
            this.notifications = Collections.EMPTY_SET;
        } else {
            this.notifications = Collections.unmodifiableSet(new HashSet(notifications));
        }
        this.priority = priority;
    }

    /**
     * Gets the source class from which this GBeanInfo can be retrieved using GBeanInfo.getGBeanInfo(className, classLoader).
     * A null source class means the gbean info was dynamically generated.
     *
     * @return the source of this GBeanInfo, or null if it was dynamically generated
     */
    public String getSourceClass() {
        return sourceClass;
    }

    public String getName() {
        return name;
    }

    public String getClassName() {
        return className;
    }

    public String getJ2eeType() {
        return j2eeType;
    }

    /**
     * Gets the info for the specified attribute, or null if there is no such
     * attribute.  Note that the attribute may have a getter or setter or both;
     * being an attribute does not imply that both methods are available.
     */
    public GAttributeInfo getAttribute(String name) {
        return (GAttributeInfo) attributesByName.get(name);
    }

    /**
     * Returns a Set where the elements are type GAttributeInfo
     */
    public Set getAttributes() {
        return attributes;
    }

    /**
     * Returns a list where the elements are type GAttributeInfo
     */
    public List getPersistentAttributes() {
        List attrs = new ArrayList();
        for (Iterator i = attributes.iterator(); i.hasNext();) {
            GAttributeInfo info = (GAttributeInfo) i.next();
            if (info.isPersistent()) {
                attrs.add(info);
            }
        }
        return attrs;
    }

    /**
     * Returns a list where the elements are type GAttributeInfo
     */
    public List getManageableAttributes() {
        List attrs = new ArrayList();
        for (Iterator i = attributes.iterator(); i.hasNext();) {
            GAttributeInfo info = (GAttributeInfo) i.next();
            if (info.isManageable()) {
                attrs.add(info);
            }
        }
        return attrs;
    }

    public GConstructorInfo getConstructor() {
        return constructor;
    }

    public Set getOperations() {
        return operations;
    }

    public Set getNotifications() {
        return notifications;
    }

    public Set getReferences() {
        return references;
    }

    public GReferenceInfo getReference(String name) {
        return (GReferenceInfo) referencesByName.get(name);
    }

    public Set getInterfaces() {
        return interfaces;
    }

    public int getPriority() {
        return priority;
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        priority = GBeanInfo.PRIORITY_NORMAL;
        in.defaultReadObject();
    }
    
    public String toString() {
        StringBuffer result = new StringBuffer("[GBeanInfo:");
        result.append(" id=").append(super.toString());
        result.append(" sourceClass=").append(sourceClass);
        result.append(" name=").append(name);
        for (Iterator iterator = attributes.iterator(); iterator.hasNext();) {
            GAttributeInfo geronimoAttributeInfo = (GAttributeInfo) iterator.next();
            result.append("\n    attribute: ").append(geronimoAttributeInfo);
        }
        for (Iterator iterator = operations.iterator(); iterator.hasNext();) {
            GOperationInfo geronimoOperationInfo = (GOperationInfo) iterator.next();
            result.append("\n    operation: ").append(geronimoOperationInfo);
        }
        for (Iterator iterator = references.iterator(); iterator.hasNext();) {
            GReferenceInfo referenceInfo = (GReferenceInfo) iterator.next();
            result.append("\n    reference: ").append(referenceInfo);
        }
        result.append("]");
        return result.toString();
    }
}