/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file;

import alluxio.client.ReadType;
import alluxio.client.block.AlluxioBlockStore;
import alluxio.client.block.BlockInStream;
import alluxio.client.block.BufferedBlockInStream;
import alluxio.client.block.TestBufferedBlockInStream;
import alluxio.client.block.TestBufferedBlockOutStream;
import alluxio.client.file.FileInStream;
import alluxio.client.file.FileSystemContext;
import alluxio.client.file.URIStatus;
import alluxio.client.file.options.InStreamOptions;
import alluxio.client.file.policy.FileWriteLocationPolicy;
import alluxio.client.file.policy.LocalFirstPolicy;
import alluxio.client.file.policy.RoundRobinPolicy;
import alluxio.client.util.ClientMockUtils;
import alluxio.client.util.ClientTestUtils;
import alluxio.exception.ExceptionMessage;
import alluxio.underfs.UnderFileSystem;
import alluxio.util.io.BufferUtils;
import alluxio.wire.FileInfo;
import alluxio.wire.WorkerNetAddress;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.mockito.verification.VerificationMode;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.powermock.reflect.Whitebox;

@RunWith(value=PowerMockRunner.class)
@PrepareForTest(value={FileSystemContext.class, AlluxioBlockStore.class, UnderFileSystem.class})
public class FileInStreamTest {
    private static final long BLOCK_LENGTH = 100L;
    private static final long FILE_LENGTH = 350L;
    private static final long NUM_STREAMS = 4L;
    private AlluxioBlockStore mBlockStore;
    private FileSystemContext mContext;
    private FileInfo mInfo;
    private URIStatus mStatus;
    private List<TestBufferedBlockOutStream> mCacheStreams;
    private FileInStream mTestStream;

    @Before
    public void before() throws IOException {
        this.mInfo = new FileInfo().setBlockSizeBytes(100L).setLength(350L);
        ClientTestUtils.setSmallBufferSizes();
        this.mContext = (FileSystemContext)PowerMockito.mock(FileSystemContext.class);
        this.mBlockStore = (AlluxioBlockStore)PowerMockito.mock(AlluxioBlockStore.class);
        Mockito.when((Object)this.mContext.getAluxioBlockStore()).thenReturn((Object)this.mBlockStore);
        this.mCacheStreams = Lists.newArrayList();
        ArrayList blockIds = Lists.newArrayList();
        int i = 0;
        while ((long)i < 4L) {
            blockIds.add(Long.valueOf(i));
            this.mCacheStreams.add(new TestBufferedBlockOutStream(i, 100L));
            Mockito.when((Object)this.mBlockStore.getInStream((long)i)).thenAnswer((Answer)new Answer<BufferedBlockInStream>(){

                public BufferedBlockInStream answer(InvocationOnMock invocation) throws Throwable {
                    long i = (Long)invocation.getArguments()[0];
                    return new TestBufferedBlockInStream(i, (int)(i * 100L), 100L);
                }
            });
            Mockito.when((Object)this.mBlockStore.getOutStream(Mockito.eq((long)i), Mockito.anyLong(), (WorkerNetAddress)Mockito.any(WorkerNetAddress.class))).thenReturn((Object)this.mCacheStreams.get(i));
            ++i;
        }
        this.mInfo.setBlockIds((List)blockIds);
        this.mStatus = new URIStatus(this.mInfo);
        Whitebox.setInternalState(FileSystemContext.class, (String)"INSTANCE", (Object)this.mContext);
        this.mTestStream = new FileInStream(this.mStatus, InStreamOptions.defaults().setReadType(ReadType.CACHE_PROMOTE));
    }

    @After
    public void after() {
        ClientTestUtils.resetClientContext();
    }

    @Test
    public void singleByteReadTest() throws Exception {
        int i = 0;
        while ((long)i < 350L) {
            Assert.assertEquals((long)(i & 0xFF), (long)this.mTestStream.read());
            ++i;
        }
        this.verifyCacheStreams(350L);
        this.mTestStream.close();
        Assert.assertTrue((boolean)((Boolean)Whitebox.getInternalState((Object)this.mTestStream, (String)"mClosed")));
    }

    @Test
    public void readHalfFileTest() throws Exception {
        this.testReadBuffer(175);
    }

    @Test
    public void readPartialBlockTest() throws Exception {
        this.testReadBuffer(50);
    }

    @Test
    public void readBlockTest() throws Exception {
        this.testReadBuffer(100);
    }

    @Test
    public void readFileTest() throws Exception {
        this.testReadBuffer(350);
    }

    @Test
    public void readOffsetTest() throws IOException {
        int offset = 33;
        int len = 100;
        byte[] buffer = new byte[offset + len];
        byte[] expectedBuffer = new byte[offset + len];
        System.arraycopy(BufferUtils.getIncreasingByteArray((int)len), 0, expectedBuffer, offset, len);
        this.mTestStream.read(buffer, offset, len);
        Assert.assertArrayEquals((byte[])expectedBuffer, (byte[])buffer);
    }

    @Test
    public void readManyChunks() throws IOException {
        int chunksize = 10;
        Assert.assertEquals((long)0L, (long)(350L % (long)chunksize));
        byte[] buffer = new byte[chunksize];
        int offset = 0;
        int i = 0;
        while ((long)i < 350L / (long)chunksize) {
            this.mTestStream.read(buffer, 0, chunksize);
            Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)offset, (int)chunksize), (byte[])buffer);
            offset += chunksize;
            ++i;
        }
        this.verifyCacheStreams(350L);
    }

    @Test
    public void testRemaining() throws IOException {
        Assert.assertEquals((long)350L, (long)this.mTestStream.remaining());
        this.mTestStream.read();
        Assert.assertEquals((long)349L, (long)this.mTestStream.remaining());
        this.mTestStream.read(new byte[150]);
        Assert.assertEquals((long)199L, (long)this.mTestStream.remaining());
        this.mTestStream.skip(140L);
        Assert.assertEquals((long)59L, (long)this.mTestStream.remaining());
        this.mTestStream.seek(310L);
        Assert.assertEquals((long)40L, (long)this.mTestStream.remaining());
        this.mTestStream.seek(130L);
        Assert.assertEquals((long)220L, (long)this.mTestStream.remaining());
    }

    @Test
    public void testSeek() throws IOException {
        int seekAmount = 50;
        int readAmount = 200;
        byte[] buffer = new byte[readAmount];
        this.mTestStream.seek((long)seekAmount);
        this.mTestStream.read(buffer);
        Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)seekAmount, (int)readAmount), (byte[])buffer);
        Assert.assertTrue((boolean)this.mCacheStreams.get(0).isCanceled());
        Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)100, (int)100), (byte[])this.mCacheStreams.get(1).getWrittenData());
        this.mTestStream.seek((long)(seekAmount + readAmount));
        this.mTestStream.seek(310L);
        Assert.assertEquals((long)54L, (long)this.mTestStream.read());
    }

    @Test
    public void testSkip() throws IOException {
        int skipAmount = 50;
        int readAmount = 200;
        byte[] buffer = new byte[readAmount];
        this.mTestStream.skip((long)skipAmount);
        this.mTestStream.read(buffer);
        Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)skipAmount, (int)readAmount), (byte[])buffer);
        Assert.assertTrue((boolean)this.mCacheStreams.get(0).isCanceled());
        Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)100, (int)100), (byte[])this.mCacheStreams.get(1).getWrittenData());
        Assert.assertEquals((long)0L, (long)this.mTestStream.skip(0L));
        Assert.assertEquals((long)50L, (long)this.mTestStream.skip(50L));
        Assert.assertEquals((long)44L, (long)this.mTestStream.read());
    }

    @Test
    public void testPromote() throws IOException {
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)0))).promote(0L);
        this.mTestStream.read();
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)1))).promote(0L);
        this.mTestStream.read();
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)1))).promote(0L);
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)0))).promote(1L);
        this.mTestStream.read(new byte[100]);
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)1))).promote(1L);
    }

    @Test
    public void failGetInStreamTest() throws IOException {
        Mockito.when((Object)this.mBlockStore.getInStream(1L)).thenThrow(new Throwable[]{new IOException("test IOException")});
        try {
            this.mTestStream.seek(100L);
            Assert.fail((String)"block store should throw exception");
        }
        catch (IOException e) {
            Assert.assertEquals((Object)"test IOException", (Object)e.getMessage());
        }
    }

    @Test
    public void failToUnderFsTest() throws IOException {
        this.mInfo.setPersisted(true).setUfsPath("testUfsPath");
        this.mStatus = new URIStatus(this.mInfo);
        Whitebox.setInternalState(FileSystemContext.class, (String)"INSTANCE", (Object)this.mContext);
        this.mTestStream = new FileInStream(this.mStatus, InStreamOptions.defaults());
        Mockito.when((Object)this.mBlockStore.getInStream(1L)).thenThrow(new Throwable[]{new IOException("test IOException")});
        UnderFileSystem ufs = ClientMockUtils.mockUnderFileSystem((String)Mockito.eq((Object)"testUfsPath"));
        InputStream stream = (InputStream)Mockito.mock(InputStream.class);
        Mockito.when((Object)ufs.open("testUfsPath")).thenReturn((Object)stream);
        Mockito.when((Object)stream.skip(100L)).thenReturn((Object)100L);
        Mockito.when((Object)stream.skip(50L)).thenReturn((Object)50L);
        this.mTestStream.seek(150L);
        ((UnderFileSystem)Mockito.verify((Object)ufs, (VerificationMode)Mockito.times((int)1))).open("testUfsPath");
        ((InputStream)Mockito.verify((Object)stream, (VerificationMode)Mockito.times((int)1))).skip(100L);
        ((InputStream)Mockito.verify((Object)stream, (VerificationMode)Mockito.times((int)1))).skip(50L);
    }

    @Test
    public void dontCacheMidBlockSeekTest() throws IOException {
        this.mTestStream.seek(150L);
        Assert.assertFalse((boolean)((Boolean)Whitebox.getInternalState((Object)this.mTestStream, (String)"mShouldCacheCurrentBlock")));
    }

    @Test
    public void readOutOfBoundsTest() throws IOException {
        this.mTestStream.read(new byte[350]);
        Assert.assertEquals((long)-1L, (long)this.mTestStream.read());
        Assert.assertEquals((long)-1L, (long)this.mTestStream.read(new byte[10]));
    }

    @Test
    public void readBadBufferTest() throws IOException {
        try {
            this.mTestStream.read(new byte[10], 5, 6);
            Assert.fail((String)"the buffer read of invalid offset/length should fail");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)String.format("Buffer length: %s, offset: %s, len: %s", 10, 5, 6), (Object)e.getMessage());
        }
    }

    @Test
    public void seekNegativeTest() throws IOException {
        try {
            this.mTestStream.seek(-1L);
            Assert.fail((String)"seeking negative position should fail");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)String.format("Seek position is negative: %s", -1), (Object)e.getMessage());
        }
    }

    @Test
    public void seekPastEndTest() throws IOException {
        try {
            this.mTestStream.seek(351L);
            Assert.fail((String)"seeking past the end of the stream should fail");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)String.format("Seek position past end of file: %s", 351L), (Object)e.getMessage());
        }
    }

    @Test
    public void skipNegativeTest() throws IOException {
        Assert.assertEquals((long)0L, (long)this.mTestStream.skip(-10L));
    }

    @Test
    public void skipInstreamExceptionTest() throws IOException {
        long skipSize = 50L;
        BlockInStream blockInStream = (BlockInStream)Mockito.mock(BlockInStream.class);
        Whitebox.setInternalState((Object)this.mTestStream, (String)"mCurrentBlockInStream", (Object)blockInStream);
        Mockito.when((Object)blockInStream.skip(skipSize)).thenReturn((Object)0L);
        Mockito.when((Object)blockInStream.remaining()).thenReturn((Object)100L);
        try {
            this.mTestStream.skip(skipSize);
            Assert.fail((String)"skip in instream should fail");
        }
        catch (IOException e) {
            Assert.assertEquals((Object)ExceptionMessage.INSTREAM_CANNOT_SKIP.getMessage(new Object[]{skipSize}), (Object)e.getMessage());
        }
    }

    @Test
    public void locationPolicyTest() {
        this.mTestStream = new FileInStream(this.mStatus, InStreamOptions.defaults().setReadType(ReadType.CACHE_PROMOTE));
        FileWriteLocationPolicy policy = (FileWriteLocationPolicy)Whitebox.getInternalState((Object)this.mTestStream, (String)"mLocationPolicy");
        Assert.assertTrue((boolean)(policy instanceof LocalFirstPolicy));
        this.mTestStream = new FileInStream(this.mStatus, InStreamOptions.defaults().setReadType(ReadType.CACHE).setLocationPolicy((FileWriteLocationPolicy)new RoundRobinPolicy()));
        policy = (FileWriteLocationPolicy)Whitebox.getInternalState((Object)this.mTestStream, (String)"mLocationPolicy");
        Assert.assertTrue((boolean)(policy instanceof RoundRobinPolicy));
    }

    @Test
    public void missingLocationPolicyTest() {
        try {
            this.mTestStream = new FileInStream(this.mStatus, InStreamOptions.defaults().setReadType(ReadType.CACHE).setLocationPolicy(null));
        }
        catch (NullPointerException e) {
            Assert.assertEquals((Object)"The location policy is not specified", (Object)e.getMessage());
        }
    }

    @Test
    public void cacheStreamBlockSizeTest() throws Exception {
        long smallSize = 50L;
        this.mInfo.setLength(smallSize);
        this.mTestStream = new FileInStream(new URIStatus(this.mInfo), InStreamOptions.defaults().setReadType(ReadType.CACHE));
        this.mTestStream.skip(smallSize / 2L);
        this.mTestStream.read(new byte[1]);
        ((AlluxioBlockStore)Mockito.verify((Object)this.mBlockStore, (VerificationMode)Mockito.times((int)1))).getOutStream(Mockito.anyLong(), Mockito.eq((long)smallSize), (WorkerNetAddress)Mockito.any(WorkerNetAddress.class));
    }

    private void testReadBuffer(int dataRead) throws Exception {
        byte[] buffer = new byte[dataRead];
        this.mTestStream.read(buffer);
        Assert.assertArrayEquals((byte[])BufferUtils.getIncreasingByteArray((int)dataRead), (byte[])buffer);
        this.verifyCacheStreams(dataRead);
    }

    private void verifyCacheStreams(long dataRead) {
        int streamIndex = 0;
        while ((long)streamIndex < 4L) {
            TestBufferedBlockOutStream stream = this.mCacheStreams.get(streamIndex);
            byte[] data = stream.getWrittenData();
            if ((long)streamIndex * 100L > dataRead) {
                Assert.assertEquals((long)0L, (long)data.length);
            } else {
                long dataStart = (long)streamIndex * 100L;
                int i = 0;
                while ((long)i < 100L && dataStart + (long)i < dataRead) {
                    Assert.assertEquals((long)((byte)(dataStart + (long)i)), (long)data[i]);
                    ++i;
                }
            }
            ++streamIndex;
        }
    }
}

