package com.xunyi.micro.web;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.lang.Nullable;

import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Enumeration;


@Slf4j
public abstract class WebUtils {

    //, proxy-revalidate
    public static final String CACHE_CONTROL_NOCACHE = "no-cache, no-store, private, must-revalidate";
    public static final String HEADER_X_REAL_IP = "X-Real-IP";
    public static final String HEADER_X_FORWARDED_FOR = "X-Forwarded-For";
    public static final String HEADER_X_FORWARDED_HOST = "X-Forwarded-Host";
    public static final String HEADER_X_FORWARDED_PORT = "X-Forwarded-Port";
    public static final String HEADER_X_FORWARDED_PROTO = "X-Forwarded-Proto";
    public static final String HEADER_X_FORWARDED_PREFIX = "X-Forwarded-Prefix";
    public static final String HEADER_X_TRACE_ID = "X-TraceID";


    private WebUtils() {
    }

    public static String getForwardedHost(HttpServletRequest request) {
        String host = request.getHeader(HEADER_X_FORWARDED_HOST);
        return host;
    }

    public static String getHost(HttpServletRequest request) {
        String forwardedFor = getForwardedHost(request);
        if (!Strings.isNullOrEmpty(forwardedFor) && !"unknown".equalsIgnoreCase(forwardedFor)) {
            int index = forwardedFor.indexOf(',');
            return index != -1 ? forwardedFor.substring(0, index) : forwardedFor;
        } else {
            return request.getHeader("Host");
        }
    }

    public static void setNoCacheResponse(HttpServletResponse response) {
        response.setHeader("Cache-Control", CACHE_CONTROL_NOCACHE);
    }

    public static String getRealIP(HttpServletRequest request){
        return request.getHeader(HEADER_X_REAL_IP);
    }
    /**
     * 注意最外层nginx配置，避免伪造
     * X-Forwarded-For $remote_addr
     */
    public static String getIpAddress(HttpServletRequest request) {
        String forwardedFor = request.getHeader(HEADER_X_FORWARDED_FOR);
        if (!Strings.isNullOrEmpty(forwardedFor) && !"unknown".equalsIgnoreCase(forwardedFor)) {
            int index = forwardedFor.indexOf(',');
            return index != -1 ? forwardedFor.substring(0, index) : forwardedFor;
        } else {
            String realIp = WebUtils.getRealIP(request);
            if (!Strings.isNullOrEmpty(realIp) && !"unknown".equalsIgnoreCase(realIp)) {
                return realIp;
            } else {
                return request.getRemoteAddr();
            }
        }
    }

    public static void response(ServletResponse response, String contentType, String content) {
        PrintWriter out = null;
        HttpServletResponse httpResponse = (HttpServletResponse)response;
        httpResponse.setContentType(contentType);
        httpResponse.setCharacterEncoding("UTF-8");

        try {
            out = response.getWriter();
            out.print(content);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (out != null) {
                out.close();
            }

        }

    }

    public static void responseJson(ServletResponse response, String json) {
        response(response, MediaType.APPLICATION_JSON_VALUE, json);
    }

    public static void responsePlain(ServletResponse response, String text) {
        response(response, MediaType.TEXT_PLAIN_VALUE, text);
    }

    public static boolean isFromForwarded(HttpServletRequest request) {
        String xForwardedFor = request.getHeader(HEADER_X_FORWARDED_FOR);
        return !Strings.isNullOrEmpty(xForwardedFor);
    }
    public static void redirectFile(HttpServletResponse response, String pathString) {
        redirectFile(response, pathString, null);
    }
    public static void redirectFile(HttpServletResponse response, String pathString,@Nullable String fileName) {

        if (Strings.isNullOrEmpty(fileName)) {
            Path path = Paths.get(pathString);
            fileName = String.valueOf(path.getFileName());
        }

        try {
            response.setContentType("application/octet-stream;charset=utf-8");
            response.setHeader("Content-Disposition", "filename=\"" + URLEncoder.encode(fileName, StandardCharsets.UTF_8.toString()) + "\"");
        } catch (UnsupportedEncodingException ignored) {
        }
        response.setHeader("X-Accel-Redirect", pathString);
    }

    public static void logHeader(HttpServletRequest request) {
        StringBuilder logBuilder = new StringBuilder();
        Enumeration<String> names = request.getHeaderNames();

        while(names.hasMoreElements()) {
            String name = (String)names.nextElement();
            logBuilder.append(name);
            logBuilder.append(request.getHeader(name));
            logBuilder.append("\r\n");
        }

        log.info(logBuilder.toString());
    }
}
