package com.xunyi.micro.propagation.instrument.async;

import com.google.common.collect.Lists;
import com.xunyi.micro.propagation.context.CurrentContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.ProxyFactoryBean;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;

@Slf4j
public class ExecutorBeanPostProcessor implements BeanPostProcessor {

    private List<String> ignoredBeans;

    private CurrentContext currentContext;

    public ExecutorBeanPostProcessor(CurrentContext currentContext) {
        this.ignoredBeans = Lists.newArrayList();
        this.currentContext = currentContext;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (bean instanceof ContextExecutorService
            || bean instanceof ContextExecutor) {
            return bean;
        }
        if (ignoredBeans.contains(beanName)) {
            return bean;
        }
//        if (bean instanceof Executor) {
//            return wrapExecutor((Executor) bean);
//        } else
        if (bean instanceof ExecutorService) {
            return wrapExecutorService(bean);
        }
        return bean;
    }

    private Object wrapExecutorService(Object bean) {
        ExecutorService executorService = (ExecutorService) bean;
        boolean classFinal = Modifier.isFinal(bean.getClass().getModifiers());
        boolean methodFinal = anyFinalMethods(executorService, ExecutorService.class);
        boolean cglibProxy = !classFinal && !methodFinal;
        return createExecutorServiceProxy(bean, cglibProxy, executorService);
    }
    Object createExecutorServiceProxy(Object bean, boolean cglibProxy,
                                      ExecutorService executor) {
        return getProxiedObject(bean, cglibProxy, executor, () -> {
//            if (executor instanceof ScheduledExecutorService) {
//                return new TraceableScheduledExecutorService(this.beanFactory, executor);
//            }
//
//            return new TraceableExecutorService(this.beanFactory, executor);
            return new ContextExecutorService(this.currentContext, executor);
        });
    }

    private ProxyFactoryBean proxyFactoryBean(Object bean, boolean cglibProxy,
                                              Executor executor, Supplier<Executor> supplier) {
        ProxyFactoryBean factory = new ProxyFactoryBean();
        factory.setProxyTargetClass(cglibProxy);
        factory.addAdvice(
                new ExecutorMethodInterceptor<Executor>(executor, this.currentContext) {
                    @Override
                    Executor executor(CurrentContext currentContext, Executor executor) {
                        return supplier.get();
                    }
                });
        factory.setTarget(bean);
        return factory;
    }

    Object getObject(ProxyFactoryBean factory) {
        return factory.getObject();
    }

    private Object getProxiedObject(Object bean, boolean cglibProxy, Executor executor,
                                    Supplier<Executor> supplier) {
        ProxyFactoryBean factory = proxyFactoryBean(bean, cglibProxy, executor, supplier);
        try {
            return getObject(factory);
        }
        catch (Exception ex) {
            if (log.isDebugEnabled()) {
                log.debug(
                        "Exception occurred while trying to get a proxy. Will fallback to a different implementation",
                        ex);
            }
//            try {
//                if (bean instanceof ThreadPoolTaskScheduler) {
//                    if (log.isDebugEnabled()) {
//                        log.debug(
//                                "Will wrap ThreadPoolTaskScheduler in its tracing representation due to previous errors");
//                    }
//                    return createThreadPoolTaskSchedulerProxy(
//                            (ThreadPoolTaskScheduler) bean).get();
//                }
//                else if (bean instanceof ScheduledThreadPoolExecutor) {
//                    if (log.isDebugEnabled()) {
//                        log.debug(
//                                "Will wrap ScheduledThreadPoolExecutor in its tracing representation due to previous errors");
//                    }
//                    return createScheduledThreadPoolExecutorProxy(
//                            (ScheduledThreadPoolExecutor) bean).get();
//                }
//            }
//            catch (Exception ex2) {
//                if (log.isDebugEnabled()) {
//                    log.debug(
//                            "Fallback for special wrappers failed, will try the tracing representation instead",
//                            ex2);
//                }
//            }
            return supplier.get();
        }
    }
//    private Object wrapExecutor(Executor bean) {
//        boolean methodFinal = anyFinalMethods(bean, Executor.class);
//        boolean classFinal = Modifier.isFinal(bean.getClass().getModifiers());
//        boolean cglibProxy = !methodFinal && !classFinal;
//        try {
//            return createProxy(bean, cglibProxy,
//                    new ExecutorMethodInterceptor<>(executor, this.beanFactory));
//        }
//        catch (AopConfigException ex) {
//            if (cglibProxy) {
//                log.debug(
//                        "Exception occurred while trying to create a proxy, falling back to JDK proxy",
//                        ex);
//                return createProxy(bean, false,
//                        new ExecutorMethodInterceptor<>(executor, this.beanFactory));
//            }
//            throw ex;
//        }
//    }
    private static <T> boolean anyFinalMethods(T object, Class<T> iface) {
        try {
            for (Method method : ReflectionUtils.getDeclaredMethods(iface)) {
                Method m = ReflectionUtils.findMethod(object.getClass(), method.getName(),
                        method.getParameterTypes());
                if (m != null && Modifier.isFinal(m.getModifiers())) {
                    return true;
                }
            }
        }
        catch (IllegalAccessError er) {
            if (log.isDebugEnabled()) {
                log.debug("Error occurred while trying to access methods", er);
            }
            return false;
        }
        return false;
    }
}
