package com.xunyi.micro.token.support;

import com.xunyi.micro.token.XYToken;
import io.jsonwebtoken.*;
import lombok.extern.slf4j.Slf4j;
import java.security.*;
import java.time.Instant;
import java.util.Map;

/**
 * RS256 非对称算法  （公私钥）
 * HS256 对称算法  （共享密钥）
 */
@Slf4j
public class XYTokenJwt extends AbstractTokenJwt<XYToken>{
    public static final String PRIV_RESOURCE = "classpath:xy-token-private.der";
    public static final String PUB_RESOURCE = "classpath:xy-token-public.der";
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.RS256;
    private static final class CLAIM_NAMES {
        public static final String UID = "xy-uid";
        public static final String ISSUED_AT = "iss";
        public static final String EXPIRES_AT = "exp";
    }

    public static class Parser extends AbstractTokenJwtParser<XYToken> {

        public Parser() {
            this(AbstractTokenJwt.readPublicKey(PUB_RESOURCE));
        }
        public Parser(Key key) {
            super(key);
        }

        @Override
        protected XYToken read(Claims claims) {
            XYToken token = new XYToken();
            Long uid = claims.get(CLAIM_NAMES.UID, Long.class);
            Instant issuedAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.ISSUED_AT, Long.class));
            Instant expiresAt = Instant.ofEpochMilli(claims.get(CLAIM_NAMES.EXPIRES_AT, Long.class));
            token.setUid(uid);
            token.setIssuedAt(issuedAt);
            token.setExpiresAt(expiresAt);
            return token;
        }
    }

    public static class Builder extends AbstractTokenJwtBuilder<XYToken> {

        public Builder() {
            this(AbstractTokenJwt.readPrivateKey(PRIV_RESOURCE),SIGNATURE_ALGORITHM);
        }

        public Builder(Key key, SignatureAlgorithm algorithm) {
            super(key, algorithm);
        }

        @Override
        protected void write(Map<String, Object> claims, XYToken token) {
            claims.put(CLAIM_NAMES.UID, token.getUid());
            claims.put(CLAIM_NAMES.ISSUED_AT, token.getIssuedAt().toEpochMilli());
            claims.put(CLAIM_NAMES.EXPIRES_AT, token.getExpiresAt().toEpochMilli());
        }
    }

}
